# Proxy Taurus – DOCUMENTATION

Este serviço é um proxy/aggregador que expõe uma fachada HTTP em PT-BR e um WebSocket para aplicações de terceiros, conectando-se ao backend Taurus para autenticação, ordens e dados de mercado em tempo real.

- Backend: Node.js (ESM) com Express e WebSocket
- Market Data: Ably Realtime (canais derivados)
- Segurança: CORS/Allowlist de IP, Rate limit por IP no /ext/v1
- Cache: candles em memória por ativo, com limpeza periódica

## Sumário
- [Execução](#execução)
- [Variáveis de Ambiente](#variáveis-de-ambiente)
- [Health e Métricas](#health-e-métricas)
- [Fachada HTTP /ext/v1](#fachada-http-extv1)
  - [Login](#post-extev1autenticacaologin)
  - [AI Analysis](#post-extev1ai-analysis)
  - [Criar usuário](#post-extev1usuarios)
  - [Saldo](#get-extev1usuariosid_usuariosaldo)
  - [Transações](#get-extev1usuariosid_usuariotransacoes)
  - [Enviar ordem](#post-extev1ordens)
  - [Status de ordem](#get-extev1ordensid_transacao)
  - [Liquidar todas](#post-extev1ordensliquidacao)
  - [Ativos](#get-extev1ativos)
  - [Snapshot](#get-extev1ativosativosnapshot)
  - [Histórico](#get-extev1ativosativohistorico)
  - [Stream SSE](#get-extev1ativosativostream)
  - [Debug de Cache](#get-extev1debugcache)
- [WebSocket /ws](#websocket-ws)
  - [Envelope](#envelope)
  - [Métodos](#métodos)
  - [Eventos](#eventos)
- [Limites e Capacidade](#limites-e-capacidade)
- [Proxy Sessions (px_)](#proxy-sessions-px_)
- [Persistência de Cache e Pré-semente](#persistência-de-cache-e-pré-semente)
- [Liquidação Automática](#liquidação-automática)
- [Auditoria Local e Banco](#auditoria-local-e-banco)
- [Endpoints Auxiliares e Debug](#endpoints-auxiliares-e-debug)
- [Boas Práticas de Cliente](#boas-práticas-de-cliente)
- [Gerenciador de Tokens (Pool)](#gerenciador-de-tokens-pool)

---

## Execução

Requisitos: Node 18+.

1. Crie um arquivo .env com as variáveis necessárias (ver seção abaixo).
2. Instale dependências e inicie:

```bash
npm install
npm run start
```

O servidor inicia em `http://localhost:${PORT||3030}` e o WS em `ws://localhost:${PORT||3030}/ws`.

## Variáveis de Ambiente

### Básicas
- PORT=3030
- API_TOKENS=token1,token2 (obrigatório; vírgulas) | API_TOKEN=token (alias quando houver apenas um token)
- INSTANCE_COUNT=1 e INSTANCE_INDEX=0 (sharding opcional por instância/PM2)
- CACHE_LIMIT=1300 (máximo de candles em memória por ativo)
- CACHE_WINDOW_MINUTES=15 (janela de retenção em minutos; use 0 para desabilitar corte por tempo)
- HISTORY_WARM_COOLDOWN_MS=30000 e HISTORY_WARM_COUNT=200 (controle da reidratação de histórico quando o cache está curto)
- TAURUS_HISTORY_MAX=50 (limite superior de candles por leitura direta do upstream)
- HISTORY_PRESEED_SYMBOLS=EURUSD,BTCUSDT (pré-carrega histórico de símbolos no boot, de forma assíncrona)
- CACHE_PERSIST_ENABLED=true e CACHE_PERSIST_INTERVAL_MS=15000 (salva/restaura candles em `websocket/data/candles-cache.json`)

### Proxy Sessions e Credenciais
- PROXY_SESSIONS_ENABLED=true (gera tokens `px_*` no login e revalida Bearer em cada requisição)
- PROXY_ACCEPT_UPSTREAM_TOKENS=false (quando true, aceita tokens Taurus reais enviados pelo cliente)
- PROXY_SESSION_TTL_SECONDS=7200 (TTL das sessões px_ em segundos)

### Segurança e Sanitização
- ALLOWED_ORIGINS=https://app.seudominio.com,https://outro.com (controle de navegadores)
- ALLOWED_IPS=10.0.0.0/8,203.0.113.10 (whitelist server-to-server)
- INTERNAL_KEY=chave-interna-opcional (libera chamadas server-side mesmo sem Origin)
- LEGACY_HTTP_ROUTES_DISABLED=true|false (remove rotas HTTP legadas `/ws/*` quando true)
- SANITIZE_UPSTREAM_OUTPUTS=true (mascara domínios/nomes Taurus em respostas HTTP/WS)

### Streaming de Mercado
- ABLY_PARALLEL_CLIENTS=1 (número de clientes Ably paralelos por instância)
- ABLY_MAX_CHANNELS=500 (limite global de canais anexados)
- PREATTACH_THROTTLE_MS=20 (intervalo entre anexos ao usar `attachAllSymbols` no boot)

### Conexões WebSocket
- MAX_CONNECTIONS_PER_TOKEN=300 (capacidade base por token ativo do shard)
- WS_CONN_HEADROOM=0.95 (margem de segurança aplicada ao cálculo automático)
- MAX_WS_CONNECTIONS= (override manual do limite final de conexões WS)

### Pool de Tokens Taurus
- TOKEN_RPS=4 (requisições por segundo por token)
- TOKEN_MAX_INFLIGHT=8 (requisições simultâneas por token)
- TOKEN_COOLDOWN_BASE_MS=2000 (cooldown inicial aplicado após 429)
- TOKEN_MAX_RETRIES=3 (retentativas no pool antes de propagar erro)

### Rate Limit HTTP /ext
- EXT_RL_WINDOW_MS=60000 (alias RATE_LIMIT_WINDOW_MS)
- EXT_RL_MAX=300 (alias RATE_LIMIT_MAX)
- EXT_RL_STANDARD_HEADERS=true
- EXT_RL_LEGACY_HEADERS=false


## Health e Métricas

- GET /health → estado: ably, symbolsConnected, channels, wsClients, uptime
- GET /metrics → métricas detalhadas: conexões WS, mensagens, distribuição de subs, tokens, capacidade etc.
  - Campos adicionais do pool: token429, tokenCooldowns, tokenPicked, tokenRetries

## Fachada HTTP /ext/v1

Todas as respostas seguem um envelope PT-BR padronizado. Algumas rotas exigem `Authorization: Bearer <token>`.

Rate limit por IP ativo: por padrão 300 requisições/min por IP (configurável por .env). IPs na allowlist (`ALLOWED_IPS`) e chamadas com `X-Internal-Key` são isentas.

Tokens de sessão: quando o proxy opera com `PROXY_SESSIONS_ENABLED=true`, o login devolve um `px_*` que deve ser enviado como Bearer. O proxy converte internamente para o token Taurus e respeita `PROXY_SESSION_TTL_SECONDS`. Só aceite tokens upstream reais quando `PROXY_ACCEPT_UPSTREAM_TOKENS=true`.

### POST /ext/v1/autenticacao/login

Body:
```json
{ "login": "user", "senha": "pass" }
```
Resposta (200):
```json
{
  "sucesso": true,
  "token": "px_...",
  "proxy_token": "px_...",
  "usuario": { "id": 1, "nome": "...", "credito": 100.0 }
}
```

Observações:
- Quando `PROXY_SESSIONS_ENABLED=true` o campo `token` é sempre um proxy-token (`px_*`) válido tanto no HTTP quanto no WebSocket; o upstream real fica oculto.
- A chamada ao backend Taurus é feita via pool de tokens do proxy (com limites de RPS/in-flight/cooldown), transparente para o cliente.

### POST /ext/v1/ai-analysis

Body (opcional):
```json
{ "imageBase64": "...", "symbol": "EURUSD", "mode": "manual|automation" }
```
Retorna uma recomendação gerada por OpenAI (se `OPENAI_API_KEY` configurada) ou uma heurística fallback baseada nos últimos candles.

### POST /ext/v1/usuarios

Body mínimo:
```json
{ "nome": "Fulano", "email": "fulano@ex.com" }
```
Campos opcionais: `telefone`, `documento`, `login`, `senha`, `token_afiliado`.

### GET /ext/v1/usuarios/:id_usuario/saldo

Headers: `Authorization: Bearer <token>`

Resposta:
```json
{ "sucesso": true, "saldo": 1000.00, "dados": { /* Taurus */ } }
```

Observação: esta rota utiliza o token do usuário (Bearer) e não entra no pool de tokens do proxy.

### GET /ext/v1/usuarios/:id_usuario/transacoes

Headers: `Authorization: Bearer px_...`

Query string suportada:
- `date` e `enddate` (YYYY-MM-DD) para recorte de período
- `user` (opcional; quando vazio, o upstream usa o usuário associado ao Bearer)
- `page` (padrão 1)

Resposta:
```json
{ "sucesso": true, "dados": { /* payload da Taurus */ } }
```

Erros retornam envelope em PT-BR; detalhes do upstream seguem em `detalhes` quando disponíveis.

### POST /ext/v1/ordens

Headers: `Authorization: Bearer <token>`

Body:
```json
{
  "ativo": "EURUSD",
  "direcao": "compra|venda|call|put|0|1",
  "valor": 10.5,
  "expiracao": 1690000123,
  "preco_referencia": 1.23456
}
```

Observações:
- Usa Bearer do usuário (fora do pool de tokens do proxy).
- Aceita `client_id` ou `clientId` para rastrear a ordem; o valor é persistido na auditoria local e reaproveitado no agendamento de liquidação.
- Cada ordem bem-sucedida agenda automaticamente uma checagem de liquidação em `pending_settlements`, respeitando `SETTLEMENT_SAFETY_DELAY_MS` e o background worker descrito abaixo.

Idempotência: no WebSocket/transaction é suportado um campo opcional `clientId` para prevenir reenvios duplicados por conexão (cache em memória por 10 minutos).

### GET /ext/v1/ordens/:id_transacao
Headers: `Authorization: Bearer <token>`

### POST /ext/v1/ordens/liquidacao
Headers: `Authorization: Bearer <token>`

### GET /ext/v1/ativos
Retorna lista completa de ativos (objetos conforme Taurus). O carregamento periódico de símbolos usa o pool com jitter para evitar thundering herd entre instâncias.

### GET /ext/v1/ativos/:ativo/snapshot?limite=100
Retorna candles recentes do cache interno.

### GET /ext/v1/ativos/:ativo/historico?intervalo=1&quantidade=50
Retorna histórico diretamente do upstream via pool do proxy (controle de RPS/in-flight e cooldown exponencial em 429).

### GET /ext/v1/ativos/:ativo/stream
Server-Sent Events (SSE) com candles em tempo real. O primeiro evento entrega o último candle disponível.

### GET /ext/v1/debug/cache/:ativo
Retorna os candles armazenados no cache interno para o ativo informado (após sanitização).

### GET /ext/v1/debug/cache
Mostra um resumo global do cache (quantidade de símbolos, métricas de memória, timestamp da última persistência).

## WebSocket /ws

### Envelope
- Request: `{ id, method, params }`
- Resposta: `{ id, ok, result? | error? }`
- Eventos push: `{ type: "event", event: "candle", symbol, data, ts }`

Observações de capacidade e limites:
- Payload máximo por mensagem WebSocket: 16KB.
- Máximo de símbolos por conexão (subscribe): 50.
- Rate-limits finos são aplicados por método (ver `RATE_LIMITS` no código). Mensagens inválidas JSON retornam erro `INVALID_JSON`.

### Métodos
- `ping` → `{ ts }`
- `symbols` → `{ count, symbols }` (lista completa de ativos)
- `snapshot{ symbol, limit? }` → candles do cache
- `history{ symbol, interval?, countback? }` → histórico upstream
- `subscribe{ symbol, conflationMs? }` → inicia eventos `candle`
- `unsubscribe{ symbol }`
- `auth{ token }` | `login{ user, pass }` | `logout`
- `balance`
- `transaction{ expiration, amount, direction, symbol, symbol_price, clientId? }`
- `tx.status{ transactionId }`
- `settlement` (NAO È MAIS UTILIZADO)

Tokens `px_*` emitidos pelo login são aceitos diretamente em `auth`/`login`; tokens Taurus só são válidos quando `PROXY_ACCEPT_UPSTREAM_TOKENS=true`.

### Eventos
- `type: "event", event: "candle"` para cada novo candle do símbolo assinado.
- Eventos de sistema esporádicos: `type: "system"`, ex.: `ably.state`.

## Limites e Capacidade

- WS: limite de conexões por instância com base em tokens ativos e headroom.
- Subscribe: máx 50 símbolos por conexão.
- Ably: `ABLY_MAX_CHANNELS` por instância (anexo sob demanda, detach com atraso).
- HTTP /ext/v1: rate limit por IP (padrão 300/min), isento para IPs na allowlist ou com `X-Internal-Key`.

Capacidade WS por instância:
- MAX_WS_CONNECTIONS (se definido) tem precedência.
- Caso contrário: tokens_ativos_shard × MAX_CONNECTIONS_PER_TOKEN × WS_CONN_HEADROOM
  - Ex.: 10 tokens × 300 × 0.95 ≈ 2850 conexões.

## Proxy Sessions (px_)

- Ativadas por `PROXY_SESSIONS_ENABLED`; o login gera um identificador `px_*` que mapeia para o token Taurus real e carrega metadados mínimos (id, nome, saldo).
- As sessões expiram conforme `PROXY_SESSION_TTL_SECONDS` (default 2h) e são coletadas a cada 5 minutos.
- Quando `PROXY_ACCEPT_UPSTREAM_TOKENS=true`, tokens reais do upstream podem ser enviados como Bearer; caso contrário apenas `px_*` são aceitos.
- O armazenamento das sessões é reutilizado pelo worker de liquidação para obter o Bearer correto mesmo após o envio da ordem.

## Persistência de Cache e Pré-semente

- Com `CACHE_PERSIST_ENABLED=true`, o cache de candles é salvo periodicamente (controle por `CACHE_PERSIST_INTERVAL_MS`) em `websocket/data/candles-cache.json`.
- Na inicialização o proxy tenta restaurar esse arquivo antes de anexar canais Ably para reduzir tempo de aquecimento.
- `HISTORY_PRESEED_SYMBOLS` permite pré-carregar histórico para símbolos específicos, de forma não bloqueante.
- Sempre que um canal Ably é anexado e o cache está curto, o proxy dispara `fetchHistory` limitado por `HISTORY_WARM_COUNT` com cooldown `HISTORY_WARM_COOLDOWN_MS`.

## Liquidação Automática

- Após `POST /ordens` com sucesso, cada transação é agendada na tabela `pending_settlements` com atraso `SETTLEMENT_SAFETY_DELAY_MS` além do tempo de expiração estimado.
- O worker roda a cada `SETTLEMENT_POLL_INTERVAL_MS`, processa até `SETTLEMENT_MAX_BATCH` registros e executa `GET /settlement` no upstream utilizando o token armazenado (ou `px_*`).
- Falhas são reagendadas com backoff exponencial baseado em `SETTLEMENT_RETRY_BASE_MS`, limitado por `SETTLEMENT_MAX_RETRY_MS`.
- Em caso de sucesso, o status da transação é atualizado para `SETTLED` na tabela `transactions` e o payload da liquidação fica salvo para auditoria.

## Auditoria Local e Banco

- O proxy usa `better-sqlite3` em `websocket/data/app.db` com WAL habilitado.
- Tabelas: `users`, `sessions`, `transactions`, `pending_settlements` e índices auxiliares para consultas por conexão/transaction_id.
- Registra logins, abertura/fechamento de conexões WS, requests/respostas de ordens, agendamentos de liquidação e resultados posteriores.
- A API `/metrics` consome esses dados para reportar métricas como `txIdemHits`, `txIdemMiss`, tentativas de liquidação e contadores de token cooldown.

## Endpoints Auxiliares e Debug

- `POST /api/transactions`: endpoint interno para persistir pedidos de ordem recebidos por outros componentes. Campos obrigatórios: `transaction_id`, `client_id`, `symbol`, `direction`, `amount`, `symbol_price`, `expiration`.
- `GET /health`: mantém o resumo de dependências (cache, Ably, quantidade de conexões, uptime).
- `GET /metrics`: expõe métricas detalhadas, inclusive contadores do pool de tokens (`token429`, `tokenCooldowns`, `tokenRetries`, `tokenPicked`).
- `GET /ext/v1/debug/cache` e `GET /ext/v1/debug/cache/:ativo`: auxiliar para inspeção dos candles persistidos (respostas passam por `SANITIZE_UPSTREAM_OUTPUTS`).

## Boas Práticas de Cliente

- Respeite os rate limits e evite pools de conexões desnecessários.
- No WebSocket, prefira `subscribe` a polling; use `conflationMs` em cenários de rede limitada.
- Implemente retry com backoff no cliente para chamadas `history` e pontos sujeitos a 429/502.
- Nunca compartilhe seu `API_TOKEN`/`API_TOKENS`. Use o fluxo de login/autorização de usuário final.

## Gerenciador de Tokens (Pool)

O proxy utiliza um pool de tokens para todas as chamadas REST ao backend Taurus que não dependem do Bearer do usuário. O pool aplica:
- Limite de RPS por token e máximo de requisições simultâneas por token.
- Cooldown exponencial com jitter em caso de 429 (Too Many Requests).
- Seleção do melhor token disponível (menor in-flight e fora de cooldown).
- Jitter no refresh de símbolos para evitar picos coincidentes entre instâncias.

Config padrão (ajustável via variáveis de ambiente):
- `TOKEN_RPS=4`
- `TOKEN_MAX_INFLIGHT=8`
- `TOKEN_COOLDOWN_BASE_MS=2000`
- `TOKEN_MAX_RETRIES=3`

Rotas que exigem autenticação do usuário (Bearer) não usam o pool — o token do usuário é encaminhado direto.

Métricas relacionadas expostas em `/metrics`:
- token429: quantidade de respostas 429 do upstream.
- tokenCooldowns: vezes em que um token entrou em cooldown.
- tokenPicked: escolhas de token efetuadas pelo pool.
- tokenRetries: retentativas por erro de rede.
