# Integração com o novo Proxy (HTTP + WebSocket)

Este guia explica como apontar o Optimus para o novo proxy realtime/HTTP em `websocket/server.js`.

## Endereços padrão (local)
- HTTP base: `http://localhost:3030`
- WS endpoint: `ws://localhost:3030/ws`

Ajuste para o domínio/porta do seu ambiente quando necessário.

## Variáveis de ambiente sugeridas
Adicione ao `.env.local` (ou `.env`) do Optimus:

```
# HTTP
NEXT_PUBLIC_PROXY_HTTP_BASE=http://localhost:3030
# WebSocket
NEXT_PUBLIC_PROXY_WS_URL=ws://localhost:3030/ws
```

Use `process.env.NEXT_PUBLIC_PROXY_HTTP_BASE` e `process.env.NEXT_PUBLIC_PROXY_WS_URL` no código.

## Fluxo de autenticação
- HTTP: `POST /ws/login` (x-www-form-urlencoded) → retorna `{ token, id, name, credit }`.
- WS: `login({ user, pass })` (mesma conexão) → associa o token à sessão WS.

Recomendação:
- Para o app web, faça login via HTTP e guarde o token de usuário em memória/secure storage.
- Use esse token para chamadas HTTP com `Authorization: Bearer <token>`.
- Para WS, você pode: (a) iniciar sem auth apenas para mercado; (b) enviar `auth({ token })` após abrir a conexão.

## HTTP: Endpoints principais
- `POST /ws/login` – login (form-urlencoded: user, pass)
- `GET  /ws/balance` – saldo (header `Authorization: Bearer <token>`) 
- `PUT  /ws/transaction` – ordem (Bearer + form-urlencoded: expiration, amount, direction, symbol, symbol_price)
- `GET  /ws/transaction/:id` – consulta de ordem (Bearer)
- `GET  /ws/settlement` – fecha ordens (Bearer)
- `GET  /ws/symbols` – símbolos
- `GET  /ws/snapshot/:symbol` – últimos candles (cache)
- `GET  /ws/history/:symbol?interval=&countback=` – histórico

## WS: Métodos principais (JSON)
Envelope de request: `{ v:1, id, method, params? }`.

- `ping()`
- Mercado: `symbols()`, `snapshot({symbol,limit})`, `history({symbol,interval,countback})`, `subscribe({symbol,conflationMs})`, `unsubscribe({symbol})`
- Auth/Conta: `auth({token})`, `login({user,pass})`, `logout()`, `balance()`
- Trading: `transaction({expiration, amount, direction, symbol, symbol_price, clientId?})`, `tx.status({transactionId})`, `settlement()`

Eventos de resposta:
- Sucesso: `{ id, ok:true, result }`
- Erro: `{ id, ok:false, error:{ code, message } }`
- Eventos push: `{ type:'event', event:'candle', symbol, data, ts }`
- Sistema: `{ type:'system', event:'ably.state'|'backpressure', data, ts }`

## Exemplo (HTTP) – Login e Balance
```ts
// login
const res = await fetch(`${process.env.NEXT_PUBLIC_PROXY_HTTP_BASE}/ws/login`, {
  method: 'POST',
  headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
  body: new URLSearchParams({ user: login, pass })
});
const { token } = await res.json();

// balance
const balRes = await fetch(`${process.env.NEXT_PUBLIC_PROXY_HTTP_BASE}/ws/balance`, {
  headers: { Authorization: `Bearer ${token}` }
});
const balance = await balRes.json();
```

## Exemplo (WS) – Login, Subscribe e Transaction
```ts
const ws = new WebSocket(process.env.NEXT_PUBLIC_PROXY_WS_URL!);
ws.onopen = () => {
  ws.send(JSON.stringify({ id:1, v:1, method:'login', params:{ user: login, pass } }));
  ws.send(JSON.stringify({ id:2, v:1, method:'subscribe', params:{ symbol:'BTCUSDT', conflationMs:100 } }));
  // transaction (depois de calcular expiration e pegar symbol_price atual)
  ws.send(JSON.stringify({ id:3, v:1, method:'transaction', params:{ expiration, amount:'10.00', direction:'call', symbol:'BTCUSDT', symbol_price:'123.45', clientId:'opt-1' } }));
};
ws.onmessage = (ev) => {
  const msg = JSON.parse(ev.data);
  // tratar {id, ok, result} e eventos {type:'event', ...}
};
```

## Cálculo de expiration e preço
- expiration: alinhe ao próximo boundary do timeframe (ex.: 1 minuto).
  ```ts
  const step = 60; // 1min
  const now = Math.floor(Date.now()/1000);
  let expiration = Math.ceil(now/step)*step;
  if (expiration - now < 10) expiration += step; // janela de segurança
  ```
- price: use o `close` do último snapshot ou do stream subscrito.

## Boas práticas e limites
- Rate limits por método no WS (ex.: transaction 1/s e 5/min por conexão).
- Máximo de ~50 símbolos por conexão.
- Se `ws.bufferedAmount` crescer (backpressure), o servidor reduz a frequência de envio.

## Troubleshooting
- 401/"token inválido": refaça o login e reenvie Bearer.
- "expiração inválida": verifique boundary do timeframe e calcule com segurança.
- ENOTFOUND em WS: confirme `NEXT_PUBLIC_PROXY_WS_URL` e ambiente selecionado.

## Produção
- Configure `ALLOWED_ORIGINS`/`ALLOWED_IPS` no `.env` do proxy.
- Use HTTPS e WSS por trás de um reverse proxy (NGINX/Cloudflare) com sticky sessions caso escale horizontalmente.
