# 🔐 Sistema de Rotas Ofuscadas

Este sistema oculta os slugs reais das rotas da sua aplicação usando hashes criptografados.

## 📋 Mapeamento de Rotas

| Rota Real | Rota Ofuscada |
|-----------|--------------|
| `/login` | `/a7f2b9c1` |
| `/dashboard` | `/b3d4e8f2` |
| `/analysis` | `/c5e6f7a3` |
| `/ai-operation` | `/d8f9a4b2` |
| `/ai-result` | `/e2g3h5i7` |
| `/automation` | `/f4j6k8l9` |

## 🚀 Como Usar

### 1. Navegar usando o Hook

Ao invés de fazer:
```typescript
router.push('/dashboard');
```

Use:
```typescript
import { useObfuscatedRouter } from '@/hooks/use-obfuscated-router';

export default function MyComponent() {
  const router = useObfuscatedRouter();
  
  return (
    <button onClick={() => router.push('dashboard')}>
      Ir para Dashboard
    </button>
  );
}
```

### 2. Links normais

Para links HTML:
```jsx
<Link href="/b3d4e8f2">Dashboard</Link>
```

Ou importando a função helper:
```typescript
import { getObfuscatedRoute } from '@/hooks/use-obfuscated-router';

<Link href={`/${getObfuscatedRoute('dashboard')}`}>
  Dashboard
</Link>
```

### 3. No navegador

O usuário acessa:
```
http://optimus.local/b3d4e8f2
```

Mas internamente é renderizado:
```
/dashboard
```

## 🔄 Fluxo de Funcionamento

1. **Usuário acessa**: `http://optimus.local/b3d4e8f2`
2. **Apache (.htaccess)**: Reescreve para `/dashboard`
3. **Next.js (middleware)**: Valida e roteia internamente
4. **Browser**: URL permanece como `/b3d4e8f2`

## ⚙️ Modificando o Mapeamento

Para adicionar/modificar rotas, edite:
- `lib/route-mapping.ts` - Defina o mapeamento
- `.htaccess` - Adicione as regras de reescrita

Exemplo para adicionar nova rota:

**1. Em `lib/route-mapping.ts`:**
```typescript
const routeMapping = {
  // ... rotas existentes
  'nova-rota': 'g5h7i9j2',
};
```

**2. Em `.htaccess`:**
```apache
RewriteRule ^g5h7i9j2/?$ /nova-rota [QSA,L]
```

## 🛡️ Segurança

- Os hashes são aleatórios e não sequenciais
- Não há lógica criptográfica real (apenas ofuscação)
- Para segurança real, combine com autenticação
- Nunca compartilhe o `route-mapping.ts` publicamente

## 📝 Notas Importantes

- O middleware do Next.js redireciona internamente (sem recarregar)
- O Apache redireciona externamente no `.htaccess`
- Sempre use `useObfuscatedRouter()` ao navegar no frontend
- Query strings são preservadas automaticamente

## 🔧 Troubleshooting

### URLs não estão sendo ofuscadas
- Verifique se `.htaccess` está ativado no Apache
- Confirme que `mod_rewrite` está habilitado
- Reinicie o Apache: `sudo systemctl restart httpd`

### Links internos quebrados
- Use `useObfuscatedRouter()` em vez de `useRouter()`
- Verifique o mapeamento em `route-mapping.ts`

## 📚 Exemplo Completo

```typescript
'use client';

import { useObfuscatedRouter } from '@/hooks/use-obfuscated-router';

export default function Navigation() {
  const router = useObfuscatedRouter();

  return (
    <nav>
      <button onClick={() => router.push('dashboard')}>
        Dashboard
      </button>
      <button onClick={() => router.push('analysis')}>
        Análise
      </button>
      <button onClick={() => router.push('ai-operation')}>
        Operação IA
      </button>
    </nav>
  );
}
```

Pronto! Suas rotas estão completamente ofuscadas! 🎉
