"use client";

import { useState, useEffect, useCallback } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  ArrowLeft,
  TrendingUp,
  TrendingDown,
  Clock,
  Target,
  Brain,
} from "lucide-react";
import Image from "next/image";
import { PageTransition } from "@/components/page-transition";
import { useToast } from "@/components/ui/use-toast";
import { SessionExpiredModal } from "@/components/session-expired-modal";
import { useSessionExpired } from "@/hooks/use-session-expired";

const RESULT_CHECK_BUFFER_MS = 3000;
const TRANSACTION_POLL_INTERVAL_MS = 3000;
const MAX_STATUS_ATTEMPTS = 5;
const FINAL_TRANSACTION_STATUSES = new Set([
  "ganhou",
  "perdeu",
  "cancelada",
  "fechada",
  "expirada",
]);

interface AnalysisResult {
  recomendacao: string;
  probabilidade: string;
  explicacao: string;
  entrada: string; // esperado como "HH:mm:ss" em UTC
}

interface LastTransaction {
  id: string | number | null;
  symbol: string;
  directionLabel: string;
  amount: string;
  expiresAt: Date | null;
  createdAt: Date;
}

interface TransactionOutcome {
  transactionId: string | number;
  statusLabel: string;
  normalizedStatus: string;
  amount: number | null;
  returns: number | null;
  profit: number | null;
  raw: unknown;
}

export default function AIResultPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { toast } = useToast();
  const {
    sessionExpiredOpen,
    sessionExpiredMessage,
    triggerSessionExpired,
    handleSessionExpiredRedirect,
  } = useSessionExpired();

  // Estados para os dados escolhidos anteriormente
  const [symbol, setSymbol] = useState<string>("");
  const [name, setName] = useState<string>("");
  const [imageUrl, setImageUrl] = useState<string>("");
  const [model, setModel] = useState<string>("");

  const [analysisResult, setAnalysisResult] = useState<AnalysisResult | null>(
    null,
  );
  const [isLoading, setIsLoading] = useState(true);
  const [tradeAmount, setTradeAmount] = useState("10.00");
  const [isSubmittingTrade, setIsSubmittingTrade] = useState(false);
  const [currentPrice, setCurrentPrice] = useState<string | null>(null);
  const [isFetchingPrice, setIsFetchingPrice] = useState(false);
  const [lastTransaction, setLastTransaction] =
    useState<LastTransaction | null>(null);
  const [remainingSeconds, setRemainingSeconds] = useState<number | null>(null);
  const [transactionOutcome, setTransactionOutcome] =
    useState<TransactionOutcome | null>(null);
  const [isCheckingOutcome, setIsCheckingOutcome] = useState(false);
  const [transactionOutcomeError, setTransactionOutcomeError] =
    useState<string | null>(null);
  const [orderLocked, setOrderLocked] = useState(false);

  useEffect(() => {
    // 1) Tenta pegar da URL (fallback)
    const urlSymbol = searchParams.get("symbol") || "";
    const urlName = searchParams.get("name") || "";
    const urlImg = searchParams.get("img") || "";
    const urlModel = searchParams.get("model") || "";

    if (urlSymbol || urlName || urlImg || urlModel) {
      setSymbol(urlSymbol);
      setName(urlName);
      setImageUrl(urlImg);
      setModel(urlModel);
    } else {
      // 2) Caso comum: ler do localStorage (foi salvo em /ai-operation)
      const stored = localStorage.getItem("aiOperationParams");
      if (stored) {
        try {
          const { symbol, name, img, model } = JSON.parse(stored);
          setSymbol(symbol || "");
          setName(name || "");
          setImageUrl(img || "");
          setModel(model || "");
        } catch {
          // ignore parse error
        }
      }
    }

    // Resultado da análise (já estava usando localStorage)
    const result = localStorage.getItem("aiAnalysisResult");
    if (result) {
      setAnalysisResult(JSON.parse(result));
      localStorage.removeItem("aiAnalysisResult"); // Clean up
    }

    setIsLoading(false);
  }, [searchParams]);

  useEffect(() => {
    if (!symbol) return;

    let cancelled = false;

    const fetchLatestPrice = async () => {
      setIsFetchingPrice(true);
      try {
        const response = await fetch(
          `/api/chart/${symbol}?interval=1&countback=1`,
          {
            cache: "no-store",
          },
        );
        if (!response.ok)
          throw new Error(`Failed to fetch price for ${symbol}`);
        const data = await response.json();
        const candles = data?.data;
        if (!cancelled && Array.isArray(candles) && candles.length > 0) {
          const lastCandle = candles[candles.length - 1];
          const closeValue = lastCandle?.close ?? lastCandle?.Close;
          if (closeValue !== undefined && closeValue !== null) {
            setCurrentPrice(closeValue.toString());
          }
        }
      } catch (error) {
        console.error("Erro ao buscar preço atual:", error);
        if (!cancelled) setCurrentPrice(null);
      } finally {
        if (!cancelled) setIsFetchingPrice(false);
      }
    };

    fetchLatestPrice();

    return () => {
      cancelled = true;
    };
  }, [symbol]);

  // === EXIBIÇÃO DO HORÁRIO DE ENTRADA ===
  // A IA já retorna o horário em formato local (HH:mm:ss)
  // Apenas formatamos para exibição
  const formatEntradaToLocal = (entrada?: string) => {
    if (!entrada) return "";
    // Aceita "HH:mm" ou "HH:mm:ss"
    const [hhStr, mmStr, ssStr = "00"] = entrada.split(":");
    const hh = String(hhStr ?? "00").padStart(2, "0");
    const mm = String(mmStr ?? "00").padStart(2, "0");
    const ss = String(ssStr ?? "00").padStart(2, "0");

    // Retorna formatado
    return `${hh}:${mm}:${ss}`;
  };
  // =============================================================

  const isCompra = analysisResult?.recomendacao
    ?.toLowerCase()
    .includes("compra");
  const tradeDirectionLabel = isCompra ? "compra" : "venda";
  const tradeDirection = isCompra ? "call" : "put";
  const parsedTradeAmount = Number(tradeAmount.replace(",", "."));
  const isTradeAmountInvalid =
    !tradeAmount.trim() ||
    Number.isNaN(parsedTradeAmount) ||
    parsedTradeAmount <= 0;

  const parseLocaleNumber = useCallback((input: string): number => {
    const trimmed = input.replace(/\s/g, "");
    if (!trimmed) return Number.NaN;
    if (trimmed.includes(",")) {
      const sanitized = trimmed.replace(/\./g, "").replace(/,/g, ".");
      return Number(sanitized);
    }
    const sanitized = trimmed.replace(/,/g, "");
    return Number(sanitized);
  }, []);

  const formatAsBRL = (value: string | number | null | undefined) => {
    if (value === null || value === undefined) return "—";
    const raw =
      typeof value === "number"
        ? value
        : parseLocaleNumber(String(value));
    if (Number.isNaN(raw)) return String(value);
    return new Intl.NumberFormat("pt-BR", {
      style: "currency",
      currency: "BRL",
      minimumFractionDigits: 2,
    }).format(raw);
  };

  const parseMonetaryValue = useCallback(
    (value: unknown, centsValue?: unknown): number | null => {
      const normalize = (num: number) =>
        Number.isFinite(num) ? Math.round(num * 100) / 100 : null;

      if (typeof centsValue === "number") {
        return normalize(centsValue / 100);
      }
      if (typeof centsValue === "string") {
        const parsed = Number(centsValue);
        if (!Number.isNaN(parsed)) {
          return normalize(parsed / 100);
        }
      }

      if (typeof value === "number") {
        return normalize(value);
      }

      if (typeof value === "string") {
        const parsed = parseLocaleNumber(value);
        if (!Number.isNaN(parsed)) {
          return normalize(parsed);
        }
      }

      return null;
    },
    [parseLocaleNumber],
  );

  const formatProfit = (value: number | null) => {
    if (value === null) return "—";
    const absolute = formatAsBRL(Math.abs(value));
    return value >= 0 ? `+${absolute}` : `-${absolute}`;
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "ganhou":
        return "text-emerald-400";
      case "perdeu":
        return "text-red-400";
      case "cancelada":
      case "expirada":
        return "text-yellow-400";
      default:
        return "text-gray-300";
    }
  };

  const parseTransactionOutcome = useCallback(
    (payload: any, transactionId: string | number): TransactionOutcome | null => {
      if (!payload) return null;

      const transaction = payload?.transaction ?? payload;
      const statusRaw = transaction?.status ?? payload?.status;
      const normalizedStatus =
        typeof statusRaw === "string" ? statusRaw.toLowerCase() : "";

      const statusLabel =
        normalizedStatus === "ganhou"
          ? "Ganhou"
          : normalizedStatus === "perdeu"
            ? "Perdeu"
            : normalizedStatus === "cancelada"
              ? "Cancelada"
              : normalizedStatus === "fechada"
                ? "Fechada"
                : normalizedStatus === "expirada"
                  ? "Expirada"
                  : typeof statusRaw === "string"
                    ? statusRaw
                    : "—";

      const amount =
        parseMonetaryValue(transaction?.amount, transaction?.amount_cents) ??
        parseMonetaryValue(payload?.amount, payload?.amount_cents) ??
        null;

      const returnsValue =
        parseMonetaryValue(transaction?.returns, transaction?.returns_cents) ??
        parseMonetaryValue(payload?.returns, payload?.returns_cents) ??
        null;

      const profit =
        returnsValue !== null && amount !== null
          ? Math.round((returnsValue - amount) * 100) / 100
          : null;

      return {
        transactionId,
        statusLabel,
        normalizedStatus,
        amount,
        returns: returnsValue,
        profit,
        raw: payload,
      };
    },
    [parseMonetaryValue],
  );

  const parseExpiration = (rawValue: unknown): Date | null => {
    if (!rawValue) return null;
    if (typeof rawValue === "number") {
      const ms = rawValue > 1e12 ? rawValue : rawValue * 1000;
      return new Date(ms);
    }
    if (typeof rawValue === "string") {
      const trimmed = rawValue.trim();
      if (!trimmed) return null;
      const numeric = Number(trimmed);
      if (!Number.isNaN(numeric)) {
        const ms = numeric > 1e12 ? numeric : numeric * 1000;
        return new Date(ms);
      }
      const date = new Date(trimmed);
      return Number.isNaN(date.getTime()) ? null : date;
    }
    if (rawValue instanceof Date) return rawValue;
    return null;
  };

  const formatCountdown = (seconds: number | null) => {
    if (seconds === null || seconds < 0) return "—";
    const clamped = Math.max(0, seconds);
    const minutes = Math.floor(clamped / 60)
      .toString()
      .padStart(2, "0");
    const secs = (clamped % 60).toString().padStart(2, "0");
    return `${minutes}:${secs}`;
  };

  const handleExecuteTransaction = async () => {
    if (!analysisResult) return;

    const token =
      typeof window !== "undefined" ? localStorage.getItem("token") : null;
    if (!token) {
      triggerSessionExpired();
      return;
    }

    const normalizedAmount = tradeAmount.replace(",", ".").trim();
    if (
      !normalizedAmount ||
      Number.isNaN(Number(normalizedAmount)) ||
      Number(normalizedAmount) <= 0
    ) {
      toast({
        title: "Valor inválido",
        description:
          "Informe um valor maior que zero para executar a operação.",
        variant: "destructive",
      });
      return;
    }

    if (!currentPrice) {
      toast({
        title: "Preço indisponível",
        description:
          "Não foi possível obter o preço atual do ativo. Tente novamente em instantes.",
        variant: "destructive",
      });
      return;
    }

    // lock the execute button until the user clicks 'Nova Análise'
    setOrderLocked(true);
    setIsSubmittingTrade(true);
    setTransactionOutcome(null);
    setTransactionOutcomeError(null);
    setIsCheckingOutcome(false);

    try {
      const response = await fetch("/api/transaction", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${token}`,
        },
        body: JSON.stringify({
          expiration: "1",
          amount: normalizedAmount,
          direction: tradeDirection,
          symbol,
          symbolPrice: currentPrice,
        }),
      });

      if (response.status === 401) {
        triggerSessionExpired();
        return;
      }

      const data = await response.json().catch(() => null);

      if (!response.ok) {
        const message =
          data?.message ||
          data?.detail ||
          data?.error ||
          "Não foi possível criar a transação.";
        throw new Error(message);
      }

      const transactionId =
        data?.transaction_id ??
        data?.dados?.transaction_id ??
        data?.dados?.transaction?.id ??
        data?.id;

      const updatedBalance =
        data?.user_credit ??
        data?.dados?.user_credit ??
        data?.dados?.usuario?.credito ??
        null;

      const payload = data?.dados ?? data ?? {};
      const expirationRaw =
        payload?.expiration_datetime ??
        payload?.expirationDate ??
        payload?.expiration ??
        payload?.transaction?.expiration_datetime ??
        payload?.transaction?.expirationDate ??
        payload?.transaction?.expiration;
      const expiresAt = parseExpiration(expirationRaw);

      if (typeof updatedBalance === "string") {
        localStorage.setItem("userBalance", updatedBalance);
      }

      setLastTransaction({
        id: transactionId ?? null,
        symbol,
        directionLabel: tradeDirectionLabel,
        amount: normalizedAmount,
        expiresAt,
        createdAt: new Date(),
      });
      setRemainingSeconds(
        expiresAt
          ? Math.max(0, Math.ceil((expiresAt.getTime() - Date.now()) / 1000))
          : null,
      );

      toast({
        title: "Transação enviada",
        description: transactionId
          ? `Ordem ${transactionId} criada. Acompanhe o resultado no painel de transações.`
          : "Sua ordem foi enviada para a corretora.",
      });
    } catch (error) {
      console.error("Erro ao executar transação:", error);
      const description =
        error instanceof Error
          ? error.message
          : "Erro desconhecido ao enviar a ordem.";
      toast({
        title: "Falha ao enviar transação",
        description,
        variant: "destructive",
      });
    } finally {
      setIsSubmittingTrade(false);
    }
  };

  useEffect(() => {
    const expiresAt = lastTransaction?.expiresAt;
    if (!expiresAt) {
      setRemainingSeconds(null);
      return;
    }

    const update = () => {
      const diff = expiresAt.getTime() - Date.now();
      setRemainingSeconds(Math.max(0, Math.ceil(diff / 1000)));
    };

    update();
    const intervalId = window.setInterval(update, 1000);
    return () => window.clearInterval(intervalId);
  }, [lastTransaction]);

  useEffect(() => {
    // Novo fluxo de polling simplificado e robusto:
    // - Executa apenas quando houver `lastTransaction` e ainda não tivermos o outcome.
    // - Aguarda a expiração + buffer de 3s (ou apenas 3s se não houver expiração no payload).
    // - Faz até `MAX_STATUS_ATTEMPTS` tentativas com `TRANSACTION_POLL_INTERVAL_MS` entre elas.
    if (!lastTransaction) return;
    const transactionId = lastTransaction.id;
    if (transactionId === null || transactionId === undefined) return;
    if (transactionOutcome?.transactionId === transactionId) return;

    let cancelled = false;
    let controller: AbortController | null = null;

    const runChecks = async () => {
      try {
  console.debug("[AIResult] iniciar checagem de resultado para", transactionId);
  setIsCheckingOutcome(true);
  setTransactionOutcomeError(null);

        // espera pela expiração se existir
        if (lastTransaction.expiresAt) {
          const waitMs = Math.max(0, lastTransaction.expiresAt.getTime() - Date.now());
          // se ainda não expirou, aguarda até expirar
          if (waitMs > 0) {
            await new Promise<void>((resolve) => setTimeout(resolve, waitMs));
          }
          // garantir + buffer
          await new Promise<void>((resolve) => setTimeout(resolve, RESULT_CHECK_BUFFER_MS));
        } else {
          // sem expiração conhecida, aguarda apenas o buffer
          await new Promise<void>((resolve) => setTimeout(resolve, RESULT_CHECK_BUFFER_MS));
        }

        if (cancelled) return;

        controller = new AbortController();

        let attempts = 0;
        let lastData: any = null;

        while (!cancelled && attempts < MAX_STATUS_ATTEMPTS) {
          attempts += 1;
          const token = typeof window !== "undefined" ? localStorage.getItem("token") : null;
          if (!token) {
            triggerSessionExpired();
            return;
          }

          console.debug(`[AIResult] tentativa ${attempts} para transacao ${transactionId}`);
          const res = await fetch(`/api/transaction/${transactionId}`, {
            method: "GET",
            headers: { Authorization: `Bearer ${token}` },
            signal: controller.signal,
          }).catch((err) => {
            // se abort ou outra falha, será tratado abaixo
            return null as unknown as Response;
          });

          if (!res) {
            console.debug(`[AIResult] resposta nula na tentativa ${attempts}`);
            // espera e tenta novamente
          } else if (res.status === 401) {
            console.debug(`[AIResult] 401 na tentativa ${attempts}`);
            triggerSessionExpired();
            return;
          } else if (!res.ok) {
            console.debug(`[AIResult] response.ok=false status=${res.status} tentativa=${attempts}`);
            // falha na requisição (5xx/4xx) — podemos decidir re-tentar
            lastData = null;
          } else {
            lastData = await res.json().catch(() => null);
            console.debug(`[AIResult] got data tentativa=${attempts}`, lastData);
            const parsed = parseTransactionOutcome(lastData, transactionId);
            if (parsed && FINAL_TRANSACTION_STATUSES.has(parsed.normalizedStatus)) {
              console.debug(`[AIResult] status final encontrado: ${parsed.normalizedStatus}`);
              setTransactionOutcome(parsed);
              return;
            }
          }

          if (attempts < MAX_STATUS_ATTEMPTS) {
            await new Promise<void>((resolve) => setTimeout(resolve, TRANSACTION_POLL_INTERVAL_MS));
          }
        }

        // depois das tentativas, grava o que obteve (mesmo que não final)
        if (!cancelled) {
          if (lastData) {
            const parsed = parseTransactionOutcome(lastData, transactionId);
            if (parsed) setTransactionOutcome(parsed);
            else
              setTransactionOutcomeError("Resultado ainda não disponível. Tente novamente em instantes.");
          } else {
            setTransactionOutcomeError("Resultado ainda não disponível. Tente novamente em instantes.");
          }
        }
      } catch (err) {
        if (!cancelled) {
          setTransactionOutcomeError(err instanceof Error ? err.message : String(err));
        }
      } finally {
        if (!cancelled) setIsCheckingOutcome(false);
      }
    };

    // dispara em background (não bloqueia o render)
    void runChecks();

    return () => {
      cancelled = true;
      try {
        controller?.abort();
      } catch {}
    };
  }, [lastTransaction, transactionOutcome, parseTransactionOutcome, triggerSessionExpired]);

  if (isLoading) {
    return (
      <>
        <PageTransition>
          <div className="min-h-screen bg-gray-950 flex items-center justify-center">
            <div className="text-center">
              <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500 mx-auto mb-4"></div>
              <p className="text-white text-lg">Carregando análise...</p>
            </div>
          </div>
        </PageTransition>
        <SessionExpiredModal
          open={sessionExpiredOpen}
          message={sessionExpiredMessage}
          onConfirm={handleSessionExpiredRedirect}
        />
      </>
    );
  }

  if (!analysisResult) {
    return (
      <>
        <PageTransition>
          <div className="min-h-screen bg-gray-950 flex items-center justify-center">
            <div className="text-center">
              <p className="text-red-400 text-lg mb-4">
                Nenhuma análise encontrada
              </p>
              <Button
                onClick={() => router.push("/analysis")}
                className="bg-blue-600 hover:bg-blue-700"
              >
                Voltar para Análise
              </Button>
            </div>
          </div>
        </PageTransition>
        <SessionExpiredModal
          open={sessionExpiredOpen}
          message={sessionExpiredMessage}
          onConfirm={handleSessionExpiredRedirect}
        />
      </>
    );
  }

  // Usa a hora convertida para exibir ao usuário
  const entradaLocal = formatEntradaToLocal(analysisResult.entrada);

  return (
    <>
      <PageTransition>
        <div className="min-h-screen bg-gray-950">
          {/* Header */}
          <header className="border-b border-gray-800 bg-gray-900">
            <div className="container mx-auto px-4 py-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <Button
                    variant="ghost"
                    size="icon"
                    className="text-gray-400 hover:text-white hover:bg-gray-800"
                    onClick={() => router.back()}
                  >
                    <ArrowLeft className="w-5 h-5" />
                  </Button>
                  <div className="flex items-center space-x-3">
                    <div className="p-2 bg-gray-800 rounded-lg">
                      <Image
                        src="/logo-optimus.png"
                        alt="OptimusIA Logo"
                        width={32}
                        height={32}
                        className="drop-shadow-lg"
                      />
                    </div>
                    <span className="text-xl font-bold text-white">
                      OptimusIA
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </header>

          {/* Main Content */}
          <main className="container mx-auto px-4 py-4 md:py-8">
            {/* Asset Header */}
            <div className="mb-6 md:mb-8">
              <div className="flex items-center space-x-3 md:space-x-4 mb-4">
                <div className="flex items-center justify-center w-12 h-12 md:w-16 md:h-16 rounded-xl bg-gray-800 overflow-hidden">
                  {imageUrl ? (
                    <img
                      src={imageUrl || "/placeholder.svg"}
                      alt={name}
                      className="w-8 h-8 md:w-12 md:h-12"
                    />
                  ) : (
                    <span className="text-lg md:text-xl font-bold text-white">
                      {symbol.substring(0, 2)}
                    </span>
                  )}
                </div>
                <div>
                  <h1 className="text-2xl md:3xl font-bold text-white">
                    {name}
                  </h1>
                  <p className="text-gray-400 text-sm md:text-base">{symbol}</p>
                </div>
              </div>

              <div className="flex items-center space-x-2 mb-4">
                <Brain className="w-5 h-5 text-blue-400" />
                <span className="text-blue-400 font-semibold">
                  Análise com {model ? model.toUpperCase() : "—"}
                </span>
              </div>
            </div>

            {/* Analysis Result */}
            <div className="grid gap-6 md:gap-8">
              {/* Recommendation Card */}
              <Card
                className={`border-2 shadow-xl ${
                  isCompra
                    ? "bg-gradient-to-br from-green-900/50 to-green-800/30 border-green-500/50"
                    : "bg-gradient-to-br from-red-900/50 to-red-800/30 border-red-500/50"
                }`}
              >
                <CardContent className="p-6 md:p-8">
                  <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center space-x-3">
                      {isCompra ? (
                        <TrendingUp className="w-8 h-8 text-green-400" />
                      ) : (
                        <TrendingDown className="w-8 h-8 text-red-400" />
                      )}
                      <div>
                        <h2 className="text-2xl md:text-3xl font-bold text-white">
                          {analysisResult.recomendacao.toUpperCase()}
                        </h2>
                        <p className="text-gray-300">Recomendação da IA</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <div
                        className={`text-3xl md:text-4xl font-bold ${isCompra ? "text-green-400" : "text-red-400"}`}
                      >
                        {analysisResult.probabilidade}
                      </div>
                      <p className="text-gray-300 text-sm">Probabilidade</p>
                    </div>
                  </div>

                  <div className="grid md:grid-cols-2 gap-6">
                    <div className="space-y-4">
                      <div className="flex items-center space-x-2">
                        <Target className="w-5 h-5 text-blue-400" />
                        <h3 className="text-lg font-semibold text-white">
                          Explicação
                        </h3>
                      </div>
                      <p className="text-gray-300 leading-relaxed">
                        {analysisResult.explicacao}
                      </p>
                    </div>

                    <div className="space-y-4">
                      <div className="flex items-center space-x-2">
                        <Clock className="w-5 h-5 text-yellow-400" />
                        <h3 className="text-lg font-semibold text-white">
                          Horário de Entrada
                        </h3>
                      </div>
                      <div className="bg-gray-800/50 rounded-lg p-4">
                        {/* ⬇️ Usa o horário convertido para local */}
                        <p className="text-2xl font-mono font-bold text-yellow-400">
                          {entradaLocal}
                        </p>
                        <p className="text-gray-400 text-sm mt-1">
                          Horário recomendado para entrada
                        </p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gray-900/60 border-gray-800 shadow-xl">
                <CardContent className="p-6 space-y-4">
                  <div className="grid gap-4 md:grid-cols-[minmax(0,1fr)_auto] md:items-end">
                    <div className="space-y-2">
                      <Label htmlFor="tradeAmount" className="text-gray-300">
                        Valor da transação (R$)
                      </Label>
                      <Input
                        id="tradeAmount"
                        type="number"
                        min="1"
                        step="0.01"
                        value={tradeAmount}
                        onChange={(event) => setTradeAmount(event.target.value)}
                        className="bg-gray-800 border-gray-700 text-white"
                      />
                    </div>
                    <Button
                      onClick={handleExecuteTransaction}
                      disabled={
                        orderLocked ||
                        isSubmittingTrade ||
                        isTradeAmountInvalid ||
                        !currentPrice
                      }
                      className={`${
                        orderLocked
                          ? "bg-gray-600 text-gray-300 cursor-not-allowed"
                          : "bg-gradient-to-r from-emerald-500 to-teal-500 hover:from-emerald-600 hover:to-teal-600 text-white"
                      } font-semibold px-6 py-3 rounded-xl`}
                    >
                      {isSubmittingTrade
                        ? "Enviando..."
                        : `Executar ${tradeDirectionLabel}`}
                    </Button>
                  </div>
                  <div className="text-sm text-gray-400 space-y-1">
                    <p>
                      A ordem será enviada como {tradeDirectionLabel} (
                      {tradeDirection === "call" ? "CALL" : "PUT"}) para{" "}
                      {symbol}.
                    </p>
                    <p>
                      Preço de referência:{" "}
                      {isFetchingPrice
                        ? "Atualizando..."
                        : formatAsBRL(currentPrice)}
                    </p>
                  </div>
                </CardContent>
              </Card>

              {/* Action Buttons */}
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button
                  onClick={() => {
                    setOrderLocked(false);
                    router.back();
                  }}
                  className="bg-gray-700 hover:bg-gray-600 text-white font-semibold px-8 py-3 rounded-xl"
                >
                  Nova Análise
                </Button>
                <Button
                  onClick={() => router.push("/dashboard")}
                  className="bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 text-white font-semibold px-8 py-3 rounded-xl"
                >
                  Voltar ao Dashboard
                </Button>
              </div>

              <Card className="bg-gray-900/60 border-gray-800 shadow-xl">
                <CardContent className="p-6 space-y-4">
                  <h3 className="text-lg font-semibold text-white">
                    Histórico recente
                  </h3>
                  {lastTransaction ? (
                    <div className="grid gap-3 text-sm text-gray-300">
                      <div className="flex flex-wrap items-center gap-2">
                        <span className="font-medium text-white">Ordem:</span>
                        <span>{lastTransaction.id ?? "—"}</span>
                      </div>
                      <div className="flex flex-wrap items-center gap-2">
                        <span className="font-medium text-white">Ativo:</span>
                        <span>{lastTransaction.symbol}</span>
                      </div>
                      <div className="flex flex-wrap items-center gap-2">
                        <span className="font-medium text-white">Direção:</span>
                        <span className="capitalize">
                          {lastTransaction.directionLabel}
                        </span>
                      </div>
                      <div className="flex flex-wrap items-center gap-2">
                        <span className="font-medium text-white">Valor:</span>
                        <span>{formatAsBRL(lastTransaction.amount)}</span>
                      </div>
                      <div className="flex flex-wrap items-center gap-2">
                        <span className="font-medium text-white">
                          Expiração:
                        </span>
                        <span>
                          {lastTransaction.expiresAt
                            ? lastTransaction.expiresAt.toLocaleTimeString(
                                "pt-BR",
                                {
                                  hour: "2-digit",
                                  minute: "2-digit",
                                  second: "2-digit",
                                },
                              )
                            : "—"}
                        </span>
                      </div>
                      <div className="flex flex-wrap items-center gap-2">
                        <span className="font-medium text-white">
                          Tempo restante:
                        </span>
                        <span className="font-mono text-base text-emerald-400">
                          {formatCountdown(remainingSeconds)}
                        </span>
                      </div>
                      {transactionOutcome ? (
                        <>
                          <div className="flex flex-wrap items-center gap-2">
                            <span className="font-medium text-white">
                              Resultado:
                            </span>
                            <span
                              className={`font-semibold ${getStatusColor(transactionOutcome.normalizedStatus)}`}
                            >
                              {transactionOutcome.statusLabel}
                            </span>
                          </div>
                          <div className="flex flex-wrap items-center gap-2">
                            <span className="font-medium text-white">
                              Valor final:
                            </span>
                            <span>
                              {formatAsBRL(transactionOutcome.returns)}
                            </span>
                          </div>
                          <div className="flex flex-wrap items-center gap-2">
                            <span className="font-medium text-white">
                              Lucro/Prejuízo:
                            </span>
                            <span
                              className={
                                transactionOutcome.profit !== null
                                  ? transactionOutcome.profit >= 0
                                    ? "text-emerald-400"
                                    : "text-red-400"
                                  : "text-gray-300"
                              }
                            >
                              {formatProfit(transactionOutcome.profit)}
                            </span>
                          </div>
                        </>
                      ) : null}
                      {isCheckingOutcome && (
                        <div className="flex items-center gap-2">
                          <svg
                            className="animate-spin h-4 w-4 text-blue-400"
                            viewBox="0 0 24 24"
                            fill="none"
                            xmlns="http://www.w3.org/2000/svg"
                            aria-hidden="true"
                          >
                            <circle
                              className="opacity-25"
                              cx="12"
                              cy="12"
                              r="10"
                              stroke="currentColor"
                              strokeWidth="4"
                            />
                            <path
                              className="opacity-75"
                              fill="currentColor"
                              d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z"
                            />
                          </svg>
                          <p className="text-sm text-blue-400">
                            Consultando resultado final da ordem...
                          </p>
                        </div>
                      )}
                      {transactionOutcomeError && (
                        <p className="text-sm text-red-400">
                          {transactionOutcomeError}
                        </p>
                      )}
                    </div>
                  ) : (
                    <p className="text-gray-400">
                      Nenhuma operação recente registrada.
                    </p>
                  )}
                </CardContent>
              </Card>
            </div>
          </main>
        </div>
      </PageTransition>
      <SessionExpiredModal
        open={sessionExpiredOpen}
        message={sessionExpiredMessage}
        onConfirm={handleSessionExpiredRedirect}
      />
    </>
  );
}
