import { NextResponse, type NextRequest } from 'next/server'

export const runtime = 'nodejs'
export const dynamic = 'force-dynamic'
export const fetchCache = 'force-no-store'

type Mode = 'manual' | 'automation'
type Source = 'openai' | 'fallback'
type FailReason = 'no_image' | 'missing_api_key' | 'http_error' | 'no_content' | 'timeout' | 'exception'

type CallResult = {
  result: AnalysisResult
  source: Source
  reason?: FailReason
  details?: string
  rawContent?: string
}

interface AnalysisResult {
  recomendacao: string
  probabilidade?: string
  explicacao?: string
  entrada?: string // HH:mm:ss UTC
}

function utcTimePlus1Minute(): string {
  return new Date(Date.now() + 60_000).toISOString().substring(11, 19)
}

function buildFallback(mode: Mode): AnalysisResult {
  const recomendacao = Math.random() > 0.5 ? 'COMPRA' : 'VENDA'
  if (mode === 'automation') return { recomendacao }
  const prob = `${Math.floor(Math.random() * (90 - 75 + 1)) + 75}%`
  return {
    recomendacao,
    probabilidade: prob,
    explicacao: 'Análise baseada nos padrões de velas recentes e indicadores do gráfico.',
    entrada: utcTimePlus1Minute(),
  }
}

interface ParsedJsonResult {
  json: Record<string, unknown> | null
  partial: boolean
  reason?: string
}

function tryParseJsonFromContent(content: string): ParsedJsonResult {
  const attempts: string[] = []

  const attemptParse = (text: string | null, label: string): Record<string, unknown> | null => {
    if (!text) return null
    try {
      return JSON.parse(text)
    } catch (err: any) {
      attempts.push(`${label}:${err?.message ?? 'invalid'}`)
      return null
    }
  }

  const direct = attemptParse(content, 'raw')
  if (direct) return { json: direct, partial: false }
  const jsonFence = content.match(/```json\s*([\s\S]*?)\s*```/i)
  const fenced = attemptParse(jsonFence?.[1] ?? null, 'fence')
  if (fenced) return { json: fenced, partial: false }

  const firstObj = content.match(/\{[\s\S]*?\}/)
  const firstParsed = attemptParse(firstObj?.[0] ?? null, 'first-object')
  if (firstParsed) return { json: firstParsed, partial: false }

  // Heurística para extrair campos mesmo com JSON quebrado (ex.: truncado pelo modelo).
  const result: Record<string, unknown> = {}
  let matched = false

  const recomendacaoMatch = content.match(/"recomendacao"\s*:\s*"([^"\n]+)"/i)
  if (recomendacaoMatch) {
    result.recomendacao = recomendacaoMatch[1]
    matched = true
  }

  const probMatch = content.match(/"probabilidade"\s*:\s*("?)([0-9]{1,3})("?)/i)
  if (probMatch) {
    result.probabilidade = Number(probMatch[2])
    matched = true
  }

  const explicacaoMatch = content.match(/"explicacao"\s*:\s*"([\s\S]*?)"(?=\s*,\s*\"|\s*\}|$)/i)
  if (explicacaoMatch) {
    result.explicacao = explicacaoMatch[1].trim()
    matched = true
  } else {
    const explicacaoKeyIndex = content.toLowerCase().indexOf('"explicacao"')
    if (explicacaoKeyIndex >= 0) {
      const afterKey = content.slice(explicacaoKeyIndex)
      const colonIndex = afterKey.indexOf(':')
      if (colonIndex >= 0) {
        const valueSegment = afterKey.slice(colonIndex + 1)
        const quoteStart = valueSegment.indexOf('"')
        if (quoteStart >= 0) {
          const afterQuote = valueSegment.slice(quoteStart + 1)
          const closingQuote = afterQuote.indexOf('"')
          const text = closingQuote >= 0 ? afterQuote.slice(0, closingQuote) : afterQuote
          if (text.trim().length > 0) {
            result.explicacao = text.trim()
            matched = true
            attempts.push('explicacao_partial')
          }
        }
      }
    }
  }

  const entradaMatch = content.match(/"entrada"\s*:\s*"([^"\n]+)"/i)
  if (entradaMatch) {
    result.entrada = entradaMatch[1]
    matched = true
  }

  if (matched) {
    attempts.push('heuristic-success')
    return { json: result, partial: true, reason: attempts.join('|') }
  }

  return { json: null, partial: false, reason: attempts.join('|') }
}

function shouldExposeDebug(): boolean {
  return process.env.NODE_ENV !== 'production' || process.env.AI_DEBUG === '1'
}

async function callOpenAI(imageBase64: string, mode: Mode): Promise<CallResult> {
  const fallback = buildFallback(mode)

  // Prefer OPENAI_TOKEN (compat com docs), fallback para OPENAI_API_KEY
  const apiKey = process.env.OPENAI_TOKEN || process.env.OPENAI_API_KEY
  if (!apiKey) return { result: fallback, source: 'fallback', reason: 'missing_api_key' }

  // Prompt recomendado (5s candles, JSON estrito)
  const userText = [
    'You are a professional trader tasked with analyzing chart images and providing structured recommendations.',
  'IMPORTANT: You must ONLY return a valid JSON object with no additional text before or after it. Any text outside the JSON structure will cause errors.',
  'Do not write introductions, thoughts, explanations, or qualquer frase fora do JSON. Apenas retorne o objeto JSON solicitado.',
    'Analyze the chart image where candles last 5 seconds, evaluating support, resistance, and trend patterns.',
    '',
    'Return EXACTLY this JSON structure and nothing else:',
    '{',
    '  "entrada": "<HH:mm:ss>",',
    '  "recomendacao": "buy" or "sell" (choose only one - NEVER write "buy or sell"),',
    '  "probabilidade": number between 60-90,',
    '  "explicacao": "A technical explanation in Portuguese based on the market written in exactly 3 sentences, dont be verbose, dont use time in explanation each concise and not exceeding 100 words"',
    '}',
    '',
    'Rules:',
    '- The "entrada" must be the next exact minute relative to current Brasília time.',
    '- The "recomendacao" must be either "buy" or "sell" based on your analysis.',
    '- The "probabilidade" must be a number between 60-90.',
    '- The "explicacao" must be in Portuguese, exactly two sentences, each with at most 30 words.',
    '- Do not include any outras frases fora do JSON. Output only the JSON object above.',
  ].join('\n')

  const baseTimeoutMs = Math.max(1000, Number(process.env.OPENAI_TIMEOUT_MS || 8000))
  try {
    const base = (process.env.OPENAI_BASE_URL || 'https://api.openai.com/v1').replace(/\/$/, '')
    const path = (process.env.OPENAI_CHAT_COMPLETIONS_PATH || '/chat/completions')
      .replace(/^\/?/, '/')
    const url = `${base}${path}`

    const headers: Record<string, string> = { 'Content-Type': 'application/json' }
    const headerName = process.env.OPENAI_AUTH_HEADER_NAME || 'Authorization'
    const scheme = (process.env.OPENAI_AUTH_SCHEME ?? 'Bearer').trim()
    // Suporta tanto Authorization: Bearer KEY quanto cabeçalhos customizados (ex.: x-api-key: KEY)
    if (headerName) {
      headers[headerName] = scheme && scheme.toLowerCase() !== 'none' ? `${scheme} ${apiKey}` : apiKey
    }
    // Headers extras opcionais via JSON (ex.: {"x-tenant":"abc"})
    if (process.env.OPENAI_EXTRA_HEADERS_JSON) {
      try {
        const extra = JSON.parse(process.env.OPENAI_EXTRA_HEADERS_JSON)
        if (extra && typeof extra === 'object') {
          for (const [k, v] of Object.entries(extra)) {
            if (typeof v === 'string') headers[k] = v
          }
        }
      } catch {}
    }

    const configuredModelRaw = process.env.OPENAI_MODEL
    const configuredList = configuredModelRaw && configuredModelRaw.trim().length > 0
      ? configuredModelRaw.split(',').map(s => s.trim()).filter(Boolean)
      : []
    const candidates = configuredList.length > 0
      ? configuredList
      : (mode === 'automation'
          ? ['llama3.2-vision:11b-instruct']
          : ['llama3.2-vision:11b-instruct'])
    const maxTokensAutomation = Number(process.env.OPENAI_MAX_TOKENS_AUTOMATION || 120)
    const maxTokensManual = Number(process.env.OPENAI_MAX_TOKENS || 400)
    const temperature = Number(process.env.OPENAI_TEMPERATURE || 0.0)
    const responseFormat = process.env.OPENAI_RESPONSE_FORMAT_JSON === '1' ? { type: 'json_object' as const } : undefined
    
    // Timeout baseado APENAS no .env.local
    const globalTimeoutMs = baseTimeoutMs
    const globalStartTime = Date.now()
    
    // Tenta modelos em sequência até um aceitar
    for (const model of candidates) {
      // Verifica se não ultrapassou o timeout global
      const elapsed = Date.now() - globalStartTime
      if (elapsed >= globalTimeoutMs) {
        if (shouldExposeDebug()) console.warn('[ai-api] Global timeout exceeded', { elapsed, limit: globalTimeoutMs, model })
        return { result: fallback, source: 'fallback', reason: 'timeout', details: `global_timeout_exceeded:${elapsed}ms` }
      }

      // Timeout para esta tentativa específica de modelo
      const remainingMs = Math.max(1000, globalTimeoutMs - elapsed)
      let timeoutHandle: NodeJS.Timeout | null = null
      let requestCompleted = false

      try {
        // Cria novo controller para cada tentativa (não reutiliza entre retry)
        const controller = new AbortController()
        
        const doCall = async (opts: { includeResponseFormat: boolean }) => fetch(url, {
          method: 'POST',
          headers,
          body: JSON.stringify({
            model,
            messages: [
              { role: 'system', content: 'You are a professional trader. Return only a valid JSON object.' },
              { role: 'user', content: [
                  { type: 'text', text: userText },
                  { type: 'image_url', image_url: { url: imageBase64 } },
                ]
              },
            ],
            max_tokens: mode === 'automation' ? maxTokensAutomation : maxTokensManual,
            temperature,
            ...(opts.includeResponseFormat && responseFormat ? { response_format: responseFormat } : {}),
          }),
          signal: controller.signal,
        })

        // Configura timeout que só ativa SE a resposta não chegar a tempo
        timeoutHandle = setTimeout(() => {
          if (!requestCompleted) {
            if (shouldExposeDebug()) console.warn('[ai-api] Request timeout firing', { remainingMs, model })
            controller.abort()
          }
        }, remainingMs)

        // 1ª tentativa: com response_format (se habilitado). Se der 5xx/429, tenta novamente sem response_format.
        let resp = await doCall({ includeResponseFormat: true })
        requestCompleted = true
        if (timeoutHandle) clearTimeout(timeoutHandle)
        
        if (!resp.ok && (resp.status >= 500 || resp.status === 429)) {
          try {
            // Limpa timeout anterior antes de tentar novamente
            if (timeoutHandle) clearTimeout(timeoutHandle)
            
            // Cria novo controller para a retry
            const retryController = new AbortController()
            const retryDoCall = async (opts: { includeResponseFormat: boolean }) => fetch(url, {
              method: 'POST',
              headers,
              body: JSON.stringify({
                model,
                messages: [
                  { role: 'system', content: 'You are a professional trader. Return only a valid JSON object.' },
                  { role: 'user', content: [
                      { type: 'text', text: userText },
                      { type: 'image_url', image_url: { url: imageBase64 } },
                    ]
                  },
                ],
                max_tokens: mode === 'automation' ? maxTokensAutomation : maxTokensManual,
                temperature,
                ...(opts.includeResponseFormat && responseFormat ? { response_format: responseFormat } : {}),
              }),
              signal: retryController.signal,
            })
            
            requestCompleted = false
            
            // Configura novo timeout para a retry
            timeoutHandle = setTimeout(() => {
              if (!requestCompleted) {
                if (shouldExposeDebug()) console.warn('[ai-api] Retry request timeout firing', { remainingMs, model })
                retryController.abort()
              }
            }, remainingMs)
            
            resp = await retryDoCall({ includeResponseFormat: false })
            requestCompleted = true
            if (timeoutHandle) clearTimeout(timeoutHandle)
          } catch {}
        }
        
        if (!resp.ok) {
          let body = ''
          try { body = await resp.text() } catch {}
          const isModelMissing = resp.status === 404 || /does not exist|not exist|model .* not found/i.test(body)
          if (shouldExposeDebug()) console.warn('[ai-api] OpenAI HTTP error', resp.status, body?.slice?.(0, 200), 'model=', model)
          // Em modelo ausente OU erros transitórios (5xx/429), tenta o próximo candidato
          if (isModelMissing || resp.status >= 500 || resp.status === 429) {
            continue
          }
          return { result: fallback, source: 'fallback', reason: 'http_error', details: `status=${resp.status}` }
        }
        
        const data = await resp.json()
        const content: string | undefined = data?.choices?.[0]?.message?.content
        if (!content) {
          if (shouldExposeDebug()) console.warn('[ai-api] OpenAI no content in response for model', model)
          // tenta próximo modelo
          continue
        }
        if (shouldExposeDebug()) {
          console.debug('[ai-api] Raw model content', { model, length: content.length, preview: content.slice(0, 800) })
        }
        const parsed = tryParseJsonFromContent(content)
        const j = (parsed.json ?? {}) as Record<string, unknown>
        const debugFlags: string[] = parsed.partial ? ['json_partial'] : []
        if (parsed.reason && parsed.reason.length > 0) debugFlags.push(`parse_trace(${parsed.reason})`)
        // Mapear buy/sell -> COMPRA/VENDA
        const rawRec = String(j.recomendacao ?? j.recomendação ?? '').toLowerCase()
        const recomendacao = rawRec === 'buy' ? 'COMPRA' : rawRec === 'sell' ? 'VENDA' : fallback.recomendacao
        // probabilidade pode vir número; normaliza para "NN%"
        const rawProb = (j as any).probabilidade ?? (j as any).probabilidade_percentual ?? (j as any).probability
        const probStr = typeof rawProb === 'number'
          ? `${Math.round(rawProb)}%`
          : typeof rawProb === 'string'
            ? (/^\d+$/.test(rawProb.trim()) ? `${rawProb.trim()}%` : rawProb.trim())
            : undefined
        if (!probStr) debugFlags.push('probabilidade_fallback')
        const explicacaoSource = (j as any).explicacao ?? (j as any).explicação
        const explicacao = explicacaoSource ? String(explicacaoSource) : String(fallback.explicacao ?? '')
  if (!explicacaoSource) debugFlags.push('explicacao_fallback')
  const entradaRaw = (j as any).entrada ?? ''
  if (!entradaRaw) debugFlags.push('entrada_fallback')
        const entrada = String(entradaRaw) || utcTimePlus1Minute()
        if (shouldExposeDebug()) {
          console.debug('[ai-api] Parsed fields', {
            rawJson: parsed.json,
            probStr,
            explicacaoSource,
            entrada,
            debugFlags,
          })
        }
        const result: AnalysisResult = mode === 'automation'
          ? { recomendacao }
          : {
              recomendacao,
              probabilidade: probStr || fallback.probabilidade,
              explicacao,
              entrada,
            }
        const detailsParts: string[] = []
        if (shouldExposeDebug()) {
          detailsParts.push(`url=${url}`)
          detailsParts.push(`model=${model}`)
          if (debugFlags.length > 0) detailsParts.push(`flags=${debugFlags.join('|')}`)
        }
        if (shouldExposeDebug() && debugFlags.length > 0) {
          console.info('[ai-api] Parsed response with fallback fields', { debugFlags, content: content.slice(0, 500) })
        }
        if (shouldExposeDebug()) {
          console.debug('[ai-api] Returning AI result', { source: 'openai', result })
        }
        return {
          result,
          source: 'openai',
          details: detailsParts.length > 0 ? detailsParts.join(';') : undefined,
          rawContent: shouldExposeDebug() ? content : undefined,
        }
      } catch (e: any) {
        // Limpa o timeout em caso de erro
        if (timeoutHandle) clearTimeout(timeoutHandle)
        
        // Erro por tentativa: registra e segue para o próximo modelo
        const isAbort = e?.name === 'AbortError'
        const isTimeout = isAbort && !requestCompleted
        if (shouldExposeDebug()) console.warn('[ai-api] OpenAI attempt error', isTimeout ? 'timeout' : (isAbort ? 'aborted' : (e?.message || e)), 'model=', model)
        
        if (isTimeout) {
          return { result: fallback, source: 'fallback', reason: 'timeout', details: `model_timeout:${remainingMs}ms:${model}` }
        }
        continue
      }
    }
    return { result: fallback, source: 'fallback', reason: 'http_error', details: 'no_model_accepted' }
  } catch (e: any) {
    const isAbort = e?.name === 'AbortError'
    if (shouldExposeDebug()) console.error('[ai-api] OpenAI exception', isAbort ? 'timeout' : e?.message || e)
    return { result: fallback, source: 'fallback', reason: isAbort ? 'timeout' : 'exception', details: shouldExposeDebug() ? (isAbort ? `timeoutMs=${baseTimeoutMs}` : (e?.message || 'error')) : undefined }
  }
}

function getCorsHeaders(origin: string | null): HeadersInit {
  const allowedRaw = process.env.AI_ALLOWED_ORIGINS || process.env.AI_CORS_ORIGINS || ''
  const allowed = allowedRaw.split(',').map((s: string) => s.trim()).filter(Boolean)
  const headers: Record<string, string> = {
    'Access-Control-Allow-Methods': 'POST,OPTIONS',
    'Access-Control-Allow-Headers': 'Content-Type, Authorization',
    'Vary': 'Origin',
  }
  if (allowed.length > 0 && origin && allowed.includes(origin)) headers['Access-Control-Allow-Origin'] = origin
  if (allowed.length === 0) headers['Access-Control-Allow-Origin'] = '*'
  return headers
}

function withExtraHeaders(base: HeadersInit, extra?: Record<string, string>): Headers {
  const h = new Headers(base)
  if (extra) {
    for (const [k, v] of Object.entries(extra)) h.set(k, v)
  }
  return h
}

export async function OPTIONS(request: NextRequest) {
  const headers = getCorsHeaders(request.headers.get('origin'))
  return new NextResponse(null, { status: 204, headers })
}

export async function POST(request: NextRequest) {
  const origin = request.headers.get('origin')
  const headers = getCorsHeaders(origin)
  try {
    let body: { imageBase64?: string; mode?: string } = {}
    try {
      body = await request.json()
    } catch {
      return NextResponse.json({ error: 'Erro ao processar a requisição' }, { status: 400, headers })
    }
    const { imageBase64, mode = 'manual' } = body
    const modeSafe: Mode = mode === 'automation' ? 'automation' : 'manual'
    if (!imageBase64) {
      const extra = shouldExposeDebug() ? { 'X-AI-Reason': 'no_image' } : undefined
      return NextResponse.json(buildFallback(modeSafe), { headers: withExtraHeaders(headers, { 'X-AI-Source': 'fallback', ...(extra || {}) }) })
    }
  const { result, source, reason, details, rawContent } = await callOpenAI(imageBase64, modeSafe)
    const extraBase = shouldExposeDebug()
      ? {
          'X-AI-Model': String(process.env.OPENAI_MODEL || (modeSafe === 'automation' ? 'gpt-4o-mini' : 'gpt-4o-mini')),
          'X-AI-Auth-Header': String(process.env.OPENAI_AUTH_HEADER_NAME || 'Authorization'),
          'X-AI-Auth-Scheme': String(process.env.OPENAI_AUTH_SCHEME ?? 'Bearer'),
          'X-AI-Timeout': String(process.env.OPENAI_TIMEOUT_MS || 8000),
          'X-AI-Endpoint': (() => { try { const u = new URL((process.env.OPENAI_BASE_URL || 'https://api.openai.com/v1').replace(/\/$/, '') + (process.env.OPENAI_CHAT_COMPLETIONS_PATH || '/chat/completions').replace(/^\/?/, '/')); return u.origin + u.pathname } catch { return 'invalid' } })(),
        }
      : undefined
    const baseHeaders = { 'X-AI-Source': source, ...(extraBase || {}) }
    const errHeaders = shouldExposeDebug() && source === 'fallback'
      ? { 'X-AI-Reason': String(reason || 'exception') }
      : undefined
    const detailHeaders = shouldExposeDebug() && details
      ? { 'X-AI-Details': String(details) }
      : undefined
    const rawHeaders = shouldExposeDebug() && rawContent
      ? { 'X-AI-Raw-Preview': rawContent.slice(0, 800).replace(/[\r\n]+/g, ' ') }
      : undefined
    return NextResponse.json(result, {
      headers: withExtraHeaders(headers, { ...baseHeaders, ...(errHeaders || {}), ...(detailHeaders || {}), ...(rawHeaders || {}) })
    })
  } catch {
    const extra = shouldExposeDebug() ? { 'X-AI-Reason': 'exception' } : undefined
    return NextResponse.json(buildFallback('manual'), { headers: withExtraHeaders(headers, { 'X-AI-Source': 'fallback', ...(extra || {}) }) })
  }
}
