// app/api/transaction/route.ts
import { type NextRequest, NextResponse } from "next/server"
import { getProxyHttpBase, withProxyHeaders } from "@/lib/proxy"

export const runtime = "nodejs"
export const dynamic = "force-dynamic"
export const fetchCache = "force-no-store"

export async function POST(request: NextRequest) {
  try {
  const { expiration, amount, direction, symbol, symbolPrice } = await request.json()

    // validação básica
    if (!expiration || !amount || !direction || !symbol || !symbolPrice) {
      return NextResponse.json({ error: "MISSING_FIELDS" }, { status: 400 })
    }

    const authHeader = request.headers.get("authorization") || request.headers.get("Authorization")
    if (!authHeader?.startsWith("Bearer ")) {
      return NextResponse.json({ error: "Token required" }, { status: 401 })
    }
    const token = authHeader.slice(7).trim()

    // Chama a fachada nova: POST /ext/v1/ordens com JSON em PT-BR
    const payload = {
      ativo: String(symbol).toUpperCase(),
      direcao: String(direction).toLowerCase(),
      valor: Number(amount),
      expiracao: String(expiration),
      preco_referencia: Number(symbolPrice),
    }
    const upstream = await fetch(`${getProxyHttpBase()}/ext/v1/ordens`, {
      method: "POST",
      headers: withProxyHeaders({
        Authorization: `Bearer ${token}`,
        "Accept": "application/json",
        "Content-Type": "application/json",
      }),
      body: JSON.stringify(payload),
      cache: "no-store",
    })

    // propaga exatamente o status/erro do provedor
    if (!upstream.ok) {
      const detailText = await upstream.text().catch(() => "")
      // retorna 401 quando o provedor devolve 401 (não transforma em 500)
      return NextResponse.json(
        {
          error: upstream.status === 401 ? "BROKER_401" : "BROKER_ERROR",
          status: upstream.status,
          detail: detailText || undefined,
        },
        { status: upstream.status },
      )
    }

    const data = await upstream.json()
    // Normaliza resposta: fachada retorna { sucesso, dados }
    if (!data?.sucesso) {
      return NextResponse.json({ error: "BROKER_ERROR", detail: data }, { status: 400 })
    }
    return NextResponse.json(data?.dados || data)
  } catch (err: any) {
    // erro inesperado local (rede, parse, etc)
    console.error("transaction route error:", err?.message || err)
    return NextResponse.json({ error: "INTERNAL", message: err?.message || "Unknown error" }, { status: 500 })
  }
}
