import { type NextRequest, NextResponse } from "next/server"
import { getProxyHttpBase, withProxyHeaders } from "@/lib/proxy"

export const runtime = "nodejs"
export const dynamic = "force-dynamic"
export const fetchCache = "force-no-store"

export async function GET(request: NextRequest) {
  try {
    const authHeader = request.headers.get("authorization")
    if (!authHeader?.startsWith("Bearer ")) {
      return NextResponse.json({ message: "Token não fornecido ou formato inválido" }, { status: 401 })
    }
    const token = authHeader.slice(7).trim()

    const url = new URL(request.url)
    const idParam = url.searchParams.get("id") || url.searchParams.get("userId") || url.searchParams.get("usuarioId")
    const userId = idParam && idParam.trim().length > 0 ? idParam.trim() : undefined

    if (!userId) {
      return NextResponse.json({ message: "Parâmetro id é obrigatório" }, { status: 400 })
    }

    const base = getProxyHttpBase()
    const upstreamUrl = `${base}/ext/v1/usuarios/${encodeURIComponent(userId)}/saldo`

    const resp = await fetch(upstreamUrl, {
      headers: withProxyHeaders({ Authorization: `Bearer ${token}`, Accept: "application/json" }),
      cache: "no-store",
    })

    if (!resp.ok) {
      const txt = await resp.text().catch(() => "")
      return NextResponse.json({ message: `Erro ao buscar saldo: ${resp.status}`, details: txt || undefined }, { status: resp.status })
    }
    const data = await resp.json()
    // Fachada retorna { sucesso, saldo, dados }
    if (!data?.sucesso) {
      return NextResponse.json({ message: data?.mensagem || "Erro ao buscar saldo" }, { status: 502 })
    }

    const saldoRaw = data?.saldo ?? data?.dados?.saldo ?? data?.dados?.credit ?? data?.dados?.credito
    let saldoNumber: number | undefined
    if (typeof saldoRaw === "number") {
      saldoNumber = saldoRaw
    } else if (typeof saldoRaw === "string") {
      const trimmed = saldoRaw.trim()
      const normalized = trimmed.includes(",") ? trimmed.replace(/\./g, "").replace(/,/g, ".") : trimmed
      const parsed = Number.parseFloat(normalized)
      if (Number.isFinite(parsed)) {
        saldoNumber = parsed
      }
    }

    const safeSaldo = saldoNumber !== undefined && Number.isFinite(saldoNumber) ? saldoNumber : 0
    const saldoFormatted = new Intl.NumberFormat("pt-BR", {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(safeSaldo)

    const creditCents = Number.isFinite(safeSaldo) ? Math.round(safeSaldo * 100) : undefined

    return NextResponse.json({
      id: data?.dados?.id ?? userId,
      credit: saldoFormatted,
      credit_cents: creditCents,
      raw: data,
    })
  } catch (error: any) {
    console.error("Erro ao buscar saldo:", error?.message || error)
    return NextResponse.json({ message: "Erro ao conectar com o servidor proxy." }, { status: 500 })
  }
}
