"use client";

import type React from "react";

import { useState, useEffect, useRef } from "react";
import { useRouter } from "next/navigation";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  ArrowLeft,
  TrendingUp,
  TrendingDown,
  Play,
  Pause,
  Zap,
  DollarSign,
  Clock,
  CheckCircle,
  XCircle,
  Target,
  RefreshCw,
} from "lucide-react";
import Image from "next/image";
import { PageTransition } from "@/components/page-transition";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";

// >>> NOVO: use o componente de gráfico
import LiveChart, {
  LiveChartHandle,
  LiveChartStats,
} from "@/components/LiveChart";
import { SessionExpiredModal } from "@/components/session-expired-modal";
import { useSessionExpired } from "@/hooks/use-session-expired";

const RESULT_CHECK_BUFFER_MS = 3000;
const TRANSACTION_POLL_INTERVAL_MS = 3000;
const FINAL_TRANSACTION_STATUSES = new Set([
  "ganhou",
  "perdeu",
  "cancelada",
  "fechada",
  "expirada",
]);

const delay = (ms: number) =>
  new Promise<void>((resolve) => {
    setTimeout(resolve, ms);
  });

interface AutomationState {
  isActive: boolean;
  currentProfit: number;
  totalTrades: number;
  winTrades: number;
  lossTrades: number;
  currentBalance: string;
  lastTransactionId: number | null;
  waitingForResult: boolean;
  lastTransactionExpirationTime: number | null; // Unix timestamp in milliseconds
}

interface TradeHistory {
  id: number;
  timestamp: string;
  direction: string;
  amount: string;
  status:
    | "Analisando..."
    | "Traçando estratégia..."
    | "Executando..."
    | "Aguardando resultado..."
    | "Ganhou"
    | "Perdeu"
    | "Cancelada"
    | "Nova oportunidade encontrada...";
  result?: string;
  price: string;
  transactionExpirationTime?: number;
}

interface WinNotification {
  id: number;
  amount: string;
  visible: boolean;
}

export default function AutomationPage() {
  const router = useRouter();
  const {
    sessionExpiredOpen,
    sessionExpiredMessage,
    triggerSessionExpired,
    handleSessionExpiredRedirect,
  } = useSessionExpired();

  // >>> ESTADOS RELACIONADOS AO GRÁFICO (vêm do componente LiveChart)
  const chartRef = useRef<LiveChartHandle>(null);
  const [currentPrice, setCurrentPrice] = useState("0");
  const [previousPrice, setPreviousPrice] = useState<string | null>(null);
  const [ohlc, setOhlc] = useState({
    open: "0",
    high: "0",
    low: "0",
    close: "0",
  });
  const [wsStatus, setWsStatus] = useState<
    "connecting" | "connected" | "disconnected"
  >("connecting");

  // Params
  const [symbol, setSymbol] = useState<string | null>(null);
  const [name, setName] = useState<string | null>(null);
  const [imageUrl, setImageUrl] = useState<string | null>(null);
  const [model, setModel] = useState<string | null>(null);

  // Automation state
  const [automation, setAutomation] = useState<AutomationState>({
    isActive: false,
    currentProfit: 0,
    totalTrades: 0,
    winTrades: 0,
    lossTrades: 0,
    currentBalance: "0,00",
    lastTransactionId: null,
    waitingForResult: false,
    lastTransactionExpirationTime: null,
  });

  // Trade history
  const [tradeHistory, setTradeHistory] = useState<TradeHistory[]>([]);

  const [entryValue, setEntryValue] = useState("5.00");
  const [galeMultiplier, setGaleMultiplier] = useState("1.1");
  const [expiration, setExpiration] = useState("1");
  const [profitTarget, setProfitTarget] = useState("100.00");
  const [stopLoss, setStopLoss] = useState("100.00");

  const [currentGaleLevel, setCurrentGaleLevel] = useState(0);
  const [currentEntryAmount, setCurrentEntryAmount] = useState("5.00");
  const [isExecutingTrade, setIsExecutingTrade] = useState(false);
  const automationActiveRef = useRef(false);
  const executingTradeRef = useRef(false);

  // Controle de estratégia: reverter entradas após 3 perdas consecutivas
  const [consecutiveLosses, setConsecutiveLosses] = useState(0);
  const [reverseTradesRemaining, setReverseTradesRemaining] = useState(0);

  const [winNotification, setWinNotification] =
    useState<WinNotification | null>(null);
  const [goalNotification, setGoalNotification] = useState<{
    visible: boolean;
    amount: string;
  } | null>(null);
  const [showInsufficientBalanceModal, setShowInsufficientBalanceModal] =
    useState(false);
  const [remainingTimes, setRemainingTimes] = useState<Record<number, number>>(
    {},
  );

  // Leave-confirm
  const [showConfirmLeaveModal, setShowConfirmLeaveModal] = useState(false);
  const [remainingTimeForTrade, setRemainingTimeForTrade] = useState(0);
  const [nextNavigationPath, setNextNavigationPath] = useState<string | null>(
    null,
  );
  const countdownIntervalRef = useRef<NodeJS.Timeout | null>(null);

  // Refresh cooldown
  const [refreshCooldown, setRefreshCooldown] = useState(0);
  const refreshIntervalRef = useRef<NodeJS.Timeout | null>(null);

  // >>> SINCRONIZAÇÃO VIA EVENTO "storage"
  useEffect(() => {
    const onStorage = (e: StorageEvent) => {
      if (e.key === "userBalance" && e.newValue) {
        setAutomation((prev) => ({ ...prev, currentBalance: e.newValue! }));
      }
    };
    window.addEventListener("storage", onStorage);
    return () => window.removeEventListener("storage", onStorage);
  }, []);

  // Balance
  const fetchUserBalance = async () => {
    const storedUser = localStorage.getItem("user");
    let parsedUser: any = null;
    if (storedUser) {
      try {
        parsedUser = JSON.parse(storedUser);
      } catch (error) {
        console.error("Erro ao analisar usuário armazenado para saldo:", error);
      }
    }

    const userIdValue = parsedUser?.id ?? parsedUser?.usuario?.id;
    const userId =
      userIdValue !== undefined && userIdValue !== null
        ? String(userIdValue)
        : null;

    if (!userId) {
      triggerSessionExpired(
        "Não foi possível validar sua sessão. Faça login novamente.",
      );
      return;
    }

    if (userId) {
      const lastRefreshTime = localStorage.getItem(`lastRefreshTime_${userId}`);
      const now = Date.now();
      const cooldownDuration = 30 * 1000;
      if (lastRefreshTime && now - Number(lastRefreshTime) < cooldownDuration) {
        const remaining = Math.ceil(
          (cooldownDuration - (now - Number(lastRefreshTime))) / 1000,
        );
        setRefreshCooldown(remaining);
        return;
      }
    }

    try {
      const token = localStorage.getItem("token");
      if (!token) {
        triggerSessionExpired();
        return;
      }
      const response = await fetch(
        `/api/user/balance?id=${encodeURIComponent(userId)}`,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "application/json",
          },
        },
      );
      if (response.ok) {
        const data = await response.json();
        const creditValue =
          typeof data.credit === "string"
            ? data.credit
            : String(data.credit ?? "0,00");
        setAutomation((prev) => ({ ...prev, currentBalance: creditValue }));
        localStorage.setItem("userBalance", creditValue);

        const nextUser = parsedUser
          ? { ...parsedUser, credit: creditValue }
          : { id: userId, credit: creditValue };
        localStorage.setItem("user", JSON.stringify(nextUser));

        if (userId) {
          localStorage.setItem(
            `lastRefreshTime_${userId}`,
            Date.now().toString(),
          );
          setRefreshCooldown(30);
          if (refreshIntervalRef.current)
            clearInterval(refreshIntervalRef.current);
          refreshIntervalRef.current = setInterval(() => {
            setRefreshCooldown((prev) => {
              if (prev <= 1) {
                if (refreshIntervalRef.current)
                  clearInterval(refreshIntervalRef.current);
                return 0;
              }
              return prev - 1;
            });
          }, 1000);
        }
      } else if (response.status === 401) {
        triggerSessionExpired();
      }
    } catch (error) {
      console.error("Error fetching balance:", error);
    }
  };

  // Main init (carrega params, saldo, cooldown)
  useEffect(() => {
    const storedParams = localStorage.getItem("aiOperationParams");
    const storedUserRaw = localStorage.getItem("user");
    let parsedStoredUser: any = null;
    if (storedUserRaw) {
      try {
        parsedStoredUser = JSON.parse(storedUserRaw);
      } catch (e) {
        console.error("Error parsing stored user for initial balance:", e);
      }
    }
    if (storedParams) {
      const params = JSON.parse(storedParams);
      setSymbol(params.symbol);
      setName(params.name);
      setImageUrl(params.img);
      setModel(params.model);

      const cachedBalance = localStorage.getItem("userBalance");
      if (cachedBalance) {
        setAutomation((prev) => ({ ...prev, currentBalance: cachedBalance }));
      } else {
        if (parsedStoredUser) {
          setAutomation((prev) => ({
            ...prev,
            currentBalance: parsedStoredUser.credit || "0,00",
          }));
        }
      }
    } else {
      router.push("/ai-operation");
    }

    // saldo + cooldown
    fetchUserBalance();
    const userIdValue = parsedStoredUser?.id ?? parsedStoredUser?.usuario?.id;
    const userId =
      userIdValue !== undefined && userIdValue !== null
        ? String(userIdValue)
        : null;
    if (userId) {
      const lastRefreshTime = localStorage.getItem(`lastRefreshTime_${userId}`);
      const now = Date.now();
      const cooldownDuration = 30 * 1000;
      if (lastRefreshTime) {
        const elapsed = now - Number(lastRefreshTime);
        if (elapsed < cooldownDuration) {
          setRefreshCooldown(Math.ceil((cooldownDuration - elapsed) / 1000));
          if (refreshIntervalRef.current)
            clearInterval(refreshIntervalRef.current);
          refreshIntervalRef.current = setInterval(() => {
            setRefreshCooldown((prev) => {
              if (prev <= 1) {
                if (refreshIntervalRef.current)
                  clearInterval(refreshIntervalRef.current);
                return 0;
              }
              return prev - 1;
            });
          }, 1000);
        }
      }
    }

    return () => {
      if (refreshIntervalRef.current) clearInterval(refreshIntervalRef.current);
    };
  }, [router]);

  // Countdown por operação (histórico)
  useEffect(() => {
    const interval = setInterval(() => {
      setRemainingTimes((prevTimes) => {
        const newTimes: Record<number, number> = {};
        tradeHistory.forEach((trade) => {
          if (
            (trade.status === "Executando..." ||
              trade.status === "Aguardando resultado...") &&
            trade.transactionExpirationTime
          ) {
            const remaining = trade.transactionExpirationTime - Date.now();
            newTimes[trade.id] = Math.max(0, Math.ceil(remaining / 1000));
          }
        });
        return newTimes;
      });
    }, 1000);
    return () => clearInterval(interval);
  }, [tradeHistory]);

  // Modal de saída — countdown
  useEffect(() => {
    if (showConfirmLeaveModal && automation.lastTransactionExpirationTime) {
      if (countdownIntervalRef.current)
        clearInterval(countdownIntervalRef.current);
      const updateCountdown = () => {
        const now = Date.now();
        const remaining = automation.lastTransactionExpirationTime! - now;
        if (remaining <= 0) {
          setRemainingTimeForTrade(0);
          if (countdownIntervalRef.current) {
            clearInterval(countdownIntervalRef.current);
            countdownIntervalRef.current = null;
          }
        } else {
          setRemainingTimeForTrade(Math.ceil(remaining / 1000));
        }
      };
      updateCountdown();
      countdownIntervalRef.current = setInterval(updateCountdown, 1000);
    } else {
      if (countdownIntervalRef.current) {
        clearInterval(countdownIntervalRef.current);
        countdownIntervalRef.current = null;
      }
    }
    return () => {
      if (countdownIntervalRef.current)
        clearInterval(countdownIntervalRef.current);
    };
  }, [showConfirmLeaveModal, automation.lastTransactionExpirationTime]);

  // Before unload
  useEffect(() => {
    const handleBeforeUnload = (event: BeforeUnloadEvent) => {
      if (automation.isActive || automation.waitingForResult) {
        event.preventDefault();
        event.returnValue = "";
        return "";
      }
    };
    window.addEventListener("beforeunload", handleBeforeUnload);
    return () => window.removeEventListener("beforeunload", handleBeforeUnload);
  }, [automation.isActive, automation.waitingForResult]);

  // ===== Helpers baseados no LiveChart =====
  const getCurrentPriceValue = () => currentPrice;

  const getPriceChangeValue = () => {
    const curr = Number.parseFloat(currentPrice || "0");
    const prev = Number.parseFloat(previousPrice || currentPrice || "0");
    const change = curr - prev;
    const percentage = prev ? (change / prev) * 100 : 0;
    return { change, percentage };
  };

  const { change, percentage } = getPriceChangeValue();
  const isPositive = change >= 0;

  // ===== Resto da lógica de automação (inalterada) =====

  const getStatusIcon = (status: TradeHistory["status"]) => {
    if (status === "Ganhou")
      return <CheckCircle className="w-4 h-4 text-green-400" />;
    if (status === "Perdeu")
      return <XCircle className="w-4 h-4 text-red-400" />;
    return <Clock className="w-4 h-4 text-yellow-400 animate-spin" />;
  };
  const getStatusColor = (status: TradeHistory["status"]) => {
    if (status === "Ganhou") return "text-green-400";
    if (status === "Perdeu") return "text-red-400";
    if (status === "Cancelada") return "text-orange-400";
    return "text-yellow-400";
  };
  const handleNavigationAttempt = (path: string) => {
    if (automation.isActive || automation.waitingForResult) {
      setNextNavigationPath(path);
      setShowConfirmLeaveModal(true);
    } else {
      router.push(path);
    }
  };

  // >>> Usa a imagem exportada pelo componente
  const exportChartAsBase64 = async (): Promise<string | null> => {
    try {
      return (await chartRef.current?.exportAsBase64()) ?? null;
    } catch {
      return null;
    }
  };

  const getAIAnalysis = async (): Promise<{ recomendacao: string }> => {
    try {
      // >>> FALLBACK FORÇADO PARA AUTOMAÇÃO <<<
      // Gera recomendação aleatória (50% COMPRA, 50% VENDA)
      const recomendacao = Math.random() > 0.5 ? "COMPRA" : "VENDA";
      return { recomendacao };
    } catch (error) {
      console.error("Error getting AI analysis:", error);
      throw error;
    }
  };

  const normalizeDirection = (dir: string) =>
    (dir || "").trim().toUpperCase();
  const invertDirection = (dir: string) => {
    const d = normalizeDirection(dir);
    if (d === "COMPRA" || d === "CALL") return "VENDA";
    if (d === "VENDA" || d === "PUT") return "COMPRA";
    return d;
  };

  // Debug opcional via localStorage.setItem('debugReverseTrades','1')
  const debugLog = (...args: any[]) => {
    try {
      const enabled =
        typeof window !== "undefined" &&
        localStorage.getItem("debugReverseTrades") === "1";
      if (enabled) console.debug("[Automation]", ...args);
    } catch {}
  };

  const calculateGaleAmount = (
    baseAmount: string,
    galeMultiplier: string,
    galeLevel: number,
  ) => {
    const base = Number.parseFloat(baseAmount);
    if (galeLevel === 0) return base;
    if (galeMultiplier === "1.1") return base * Math.pow(2, galeLevel);
    if (galeMultiplier === "1.2") return base * Math.pow(3, galeLevel);
    if (galeMultiplier === "1.3") return base * Math.pow(4, galeLevel);
    if (galeMultiplier === "1.4") return base * Math.pow(5, galeLevel);
    return base;
  };

  const createTransaction = async (
    direction: string,
    currentPrice: string,
    amount?: string,
  ) => {
    try {
      const token = localStorage.getItem("token");
      if (!token) {
        triggerSessionExpired();
        throw new Error("Token not found");
      }
      const transactionAmount = amount || entryValue;
      const response = await fetch("/api/transaction", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${token}`,
        },
        body: JSON.stringify({
          expiration,
          amount: transactionAmount,
          direction: direction === "COMPRA" ? "call" : "put",
          symbol,
          symbolPrice: currentPrice,
          token,
        }),
      });
      if (response.status === 401) {
        triggerSessionExpired();
        throw new Error("Sessão expirada");
      }
      if (!response.ok) throw new Error("Failed to create transaction");
      const data = await response.json();
      setAutomation((prev) => ({
        ...prev,
        lastTransactionId: data.transaction_id,
        waitingForResult: true,
        currentBalance: data.user_credit,
        lastTransactionExpirationTime: new Date(
          data.expiration_datetime,
        ).getTime(),
      }));
      localStorage.setItem("userBalance", data.user_credit);
      return data;
    } catch (error) {
      console.error("Error creating transaction:", error);
      throw error;
    }
  };

  const checkTransactionResult = async (transactionId: number) => {
    try {
      const token = localStorage.getItem("token");
      if (!token) {
        triggerSessionExpired();
        throw new Error("Token not found");
      }
      const response = await fetch(`/api/transaction/${transactionId}`, {
        method: "GET",
        headers: { Authorization: `Bearer ${token}` },
      });
      if (response.status === 401) {
        triggerSessionExpired();
        throw new Error("Sessão expirada");
      }
      if (!response.ok) throw new Error("Failed to check transaction result");
      return await response.json();
    } catch (error) {
      console.error("Error checking transaction result:", error);
      throw error;
    }
  };

  const waitForFinalTransactionStatus = async (transactionId: number) => {
    for (;;) {
      try {
        const result = await checkTransactionResult(transactionId);
        const status = result?.transaction?.status;
        const normalizedStatus =
          typeof status === "string" ? status.toLowerCase() : "";
        if (FINAL_TRANSACTION_STATUSES.has(normalizedStatus)) {
          return result;
        }
      } catch (error) {
        console.error("Erro consultando transação:", error);
      }
      await delay(TRANSACTION_POLL_INTERVAL_MS);
    }
  };

  const processTransactionResult = (
    result: any,
    {
      tradeId,
      galeLevel,
      baseAmount,
      currentAmount,
    }: {
      tradeId: number;
      galeLevel: number;
      baseAmount: string;
      currentAmount: number;
    },
  ) => {
    const transactionData = result?.transaction;
    if (!transactionData) {
      throw new Error("Invalid transaction payload");
    }

    const statusFromBroker: string | undefined = transactionData.status;
    const normalizedStatus =
      typeof statusFromBroker === "string"
        ? statusFromBroker.toLowerCase()
        : "";
    const rawAmountCents = transactionData.amount_cents;
    const amountCents =
      typeof rawAmountCents === "number"
        ? rawAmountCents
        : Math.round(currentAmount * 100);
    const rawReturnsCents = transactionData.returns_cents;
    const returnsCents =
      typeof rawReturnsCents === "number" ? rawReturnsCents : amountCents;
    const profitCents = returnsCents - amountCents;

    const finalStatus: TradeHistory["status"] =
      normalizedStatus === "ganhou"
        ? "Ganhou"
        : normalizedStatus === "cancelada"
          ? "Cancelada"
          : "Perdeu";

    const won = normalizedStatus === "ganhou" || profitCents > 0;

    if (won && profitCents > 0) {
      playWinSound();
      const winAmount = (profitCents / 100).toFixed(2);
      showWinNotification(`+R$ ${winAmount}`);
    }

    const profitValue = profitCents / 100;
    const profitFormatted =
      profitValue >= 0
        ? `+R$ ${profitValue.toFixed(2)}`
        : `-R$ ${Math.abs(profitValue).toFixed(2)}`;

    updateTradeInHistory(tradeId, {
      status: finalStatus,
      result: profitFormatted,
    });

    // Atualiza contagem de perdas consecutivas e ativa reversão quando necessário
    if (won) {
      setConsecutiveLosses(0);
      debugLog("Win detected. Reset consecutive losses to 0");
    } else {
      setConsecutiveLosses((prev) => {
        const next = prev + 1;
        // Ativa modo reverso para as próximas 2 entradas ao atingir 3 perdas consecutivas
        if (next >= 3 && reverseTradesRemaining === 0) {
          setReverseTradesRemaining(2);
          debugLog(
            "Activated reverse mode for next 2 trades after 3 consecutive losses",
          );
          return 0; // zera a sequência após ativar
        }
        return next;
      });
    }

    let nextGaleLevel = galeLevel;
    if (won) {
      nextGaleLevel = 0;
      setCurrentGaleLevel(0);
      setCurrentEntryAmount(entryValue);
    } else {
      nextGaleLevel = galeLevel + 1;
      const nextAmount = calculateGaleAmount(
        baseAmount,
        galeMultiplier,
        nextGaleLevel,
      );
      setCurrentGaleLevel(nextGaleLevel);
      setCurrentEntryAmount(nextAmount.toString());
    }

    const rawBalance = transactionData.user?.balance;
    if (rawBalance !== undefined && rawBalance !== null) {
      const balanceString = String(rawBalance);
      const storedUser = localStorage.getItem("user");
      if (storedUser) {
        try {
          const parsedUser = JSON.parse(storedUser);
          localStorage.setItem(
            "user",
            JSON.stringify({ ...parsedUser, credit: rawBalance }),
          );
        } catch {}
      }
      localStorage.setItem("userBalance", balanceString);
    }

    let shouldStop = false;
    setAutomation((prev) => {
      const newTotalTrades = prev.totalTrades + 1;
      const newWinTrades = won ? prev.winTrades + 1 : prev.winTrades;
      const newLossTrades = won ? prev.lossTrades : prev.lossTrades + 1;
      const profitDelta = profitValue;
      const newCurrentProfit = prev.currentProfit + profitDelta;
      const targetProfit = Number.parseFloat(profitTarget);
      const stopLossValue = Number.parseFloat(stopLoss);

      shouldStop =
        newCurrentProfit >= targetProfit || newCurrentProfit <= -stopLossValue;
      if (newCurrentProfit >= targetProfit) {
        showGoalNotification(newCurrentProfit.toFixed(2));
        automationActiveRef.current = false;
      } else if (newCurrentProfit <= -stopLossValue) {
        automationActiveRef.current = false;
      } else {
        automationActiveRef.current = true;
      }

      const balanceToPersist =
        rawBalance !== undefined && rawBalance !== null
          ? String(rawBalance)
          : prev.currentBalance;

      return {
        ...prev,
        totalTrades: newTotalTrades,
        winTrades: newWinTrades,
        lossTrades: newLossTrades,
        currentProfit: newCurrentProfit,
        waitingForResult: false,
        currentBalance: balanceToPersist,
        isActive: shouldStop ? false : prev.isActive,
        lastTransactionExpirationTime: null,
      };
    });

    return { shouldStop, nextGaleLevel };
  };

  const scheduleTransactionResultHandling = (
    transactionId: number,
    expirationDatetime: string,
    galeLevel: number,
    baseAmount: string,
    currentAmount: number,
    tradeId: number,
  ) => {
    const expirationTime = new Date(expirationDatetime).getTime();
    const timeUntilExpiration = expirationTime - Date.now();
    const waitTime = Math.max(
      timeUntilExpiration + RESULT_CHECK_BUFFER_MS,
      RESULT_CHECK_BUFFER_MS,
    );

    setTimeout(async () => {
      try {
        const result = await waitForFinalTransactionStatus(transactionId);
        const { shouldStop, nextGaleLevel } = processTransactionResult(result, {
          tradeId,
          galeLevel,
          baseAmount,
          currentAmount,
        });

        executingTradeRef.current = false;
        setIsExecutingTrade(false);

        if (!shouldStop && automationActiveRef.current) {
          setTimeout(() => {
            if (automationActiveRef.current) {
              executeAutomatedTrade(true, nextGaleLevel, baseAmount);
            }
          }, 5000);
        }
      } catch (error) {
        console.error("Erro ao verificar resultado:", error);
        updateTradeInHistory(tradeId, { status: "Perdeu", result: "Erro" });
        setAutomation((prev) => ({
          ...prev,
          waitingForResult: false,
          lastTransactionExpirationTime: null,
        }));
        executingTradeRef.current = false;
        setIsExecutingTrade(false);

        if (automationActiveRef.current) {
          setTimeout(() => {
            if (automationActiveRef.current) executeAutomatedTrade();
          }, 10000);
        }
      }
    }, waitTime);
  };

  const addTradeToHistory = (trade: Omit<TradeHistory, "id">) => {
    const newTrade: TradeHistory = { ...trade, id: Date.now() };
    setTradeHistory((prev) => [newTrade, ...prev]);
    return newTrade.id;
  };
  const updateTradeInHistory = (id: number, updates: Partial<TradeHistory>) => {
    setTradeHistory((prev) =>
      prev.map((trade) => (trade.id === id ? { ...trade, ...updates } : trade)),
    );
  };

  const playWinSound = () => {
    try {
      const audio = new Audio("/ganhou.mp3");
      audio.volume = 0.7;
      audio.play().catch(() => {});
    } catch {}
  };
  const playGoalSound = () => {
    try {
      const audio = new Audio("/meta.mp3");
      audio.volume = 0.7;
      audio.play().catch(() => {});
    } catch {}
  };
  const showWinNotification = (winAmount: string) => {
    const notification: WinNotification = {
      id: Date.now(),
      amount: winAmount,
      visible: true,
    };
    setWinNotification(notification);
    setTimeout(() => {
      setWinNotification((prev) => (prev ? { ...prev, visible: false } : null));
      setTimeout(() => setWinNotification(null), 300);
    }, 4000);
  };
  const showGoalNotification = (profitAmount: string) => {
    setGoalNotification({ visible: true, amount: profitAmount });
    playGoalSound();
  };
  const dismissGoalNotification = () => {
    setGoalNotification((prev) => (prev ? { ...prev, visible: false } : null));
    setTimeout(() => setGoalNotification(null), 300);
  };

  // ===== Execução da automação (mantida) =====
  const executeAutomatedTrade = async (
    forceActive = false,
    galeLevel = currentGaleLevel,
    baseAmount = entryValue,
  ) => {
    if (executingTradeRef.current || isExecutingTrade) return;
    if (!forceActive && !automationActiveRef.current) return;

    const currentAmount = calculateGaleAmount(
      entryValue,
      galeMultiplier,
      galeLevel,
    );
    const currentBalanceNum = Number.parseFloat(
      automation.currentBalance.replace(/\./g, "").replace(",", "."),
    );
    if (currentBalanceNum < currentAmount) {
      automationActiveRef.current = false;
      setAutomation((prev) => ({ ...prev, isActive: false }));
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      setShowInsufficientBalanceModal(true);
      return;
    }

    executingTradeRef.current = true;
    setIsExecutingTrade(true);

    const currentProfit = automation.currentProfit;
    const stopLossValue = Number.parseFloat(stopLoss);
    const potentialProfitIfLoss = currentProfit - currentAmount;
    if (potentialProfitIfLoss <= -stopLossValue) {
      automationActiveRef.current = false;
      setAutomation((prev) => ({ ...prev, isActive: false }));
      addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "STOP LOSS",
        amount: currentAmount.toFixed(2),
        status: "Cancelada",
        result: `Stop Loss Atingido: -R$ ${stopLossValue.toFixed(2)}`,
        price: getCurrentPriceValue(),
      });
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      return;
    }

    try {
      const tradeId = addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "",
        amount: currentAmount.toString(),
        status: "Analisando...",
        price: getCurrentPriceValue(),
      });
      setTimeout(
        () =>
          updateTradeInHistory(tradeId, { status: "Traçando estratégia..." }),
        1000,
      );
      setTimeout(
        () => updateTradeInHistory(tradeId, { status: "Executando..." }),
        2000,
      );

      const analysis = await getAIAnalysis();
      let finalDirection = normalizeDirection(analysis.recomendacao);
      if (reverseTradesRemaining > 0) {
        const inverted = invertDirection(finalDirection);
        debugLog("Inverting direction", {
          original: finalDirection,
          inverted,
          remainingBefore: reverseTradesRemaining,
        });
        finalDirection = inverted;
        const remainingAfter = Math.max(0, reverseTradesRemaining - 1);
        setReverseTradesRemaining(remainingAfter);
        debugLog("Reverse trades remaining", remainingAfter);
        if (remainingAfter === 0) debugLog("Reverse mode finished");
      }
      updateTradeInHistory(tradeId, {
        direction: finalDirection,
        status: "Aguardando resultado...",
      });

      const currentPriceNow = getCurrentPriceValue();
      try {
        const transaction = await createTransaction(
          finalDirection,
          currentPriceNow,
          currentAmount.toString(),
        );
        updateTradeInHistory(tradeId, {
          transactionExpirationTime: new Date(
            transaction.expiration_datetime,
          ).getTime(),
        });
        scheduleTransactionResultHandling(
          transaction.transaction_id,
          transaction.expiration_datetime,
          galeLevel,
          baseAmount,
          currentAmount,
          tradeId,
        );
      } catch (transactionError) {
        console.error("Erro ao criar transação:", transactionError);
        const errorMessage =
          transactionError instanceof Error
            ? transactionError.message
            : String(transactionError);
        if (
          errorMessage.includes("400") ||
          errorMessage.includes("Failed to create transaction")
        ) {
          updateTradeInHistory(tradeId, {
            status: "Cancelada" as any,
            result: "Erro 400 - Transação cancelada",
          });
          triggerSessionExpired(
            "Erro na transação. Sua sessão foi expirada. Faça login novamente.",
          );
        } else {
          updateTradeInHistory(tradeId, {
            status: "Perdeu",
            result: "Nova oportunidade Encontrada",
          });
        }
        setAutomation((prev) => ({
          ...prev,
          waitingForResult: false,
          lastTransactionExpirationTime: null,
        }));
        executingTradeRef.current = false;
        setIsExecutingTrade(false);
        if (automationActiveRef.current) {
          setTimeout(() => {
            if (automationActiveRef.current)
              executeAutomatedTrade(true, galeLevel, baseAmount);
          }, 5000);
        }
        return;
      }
    } catch (error) {
      console.error("Erro no trade automatizado:", error);
      const currentAmount = calculateGaleAmount(
        baseAmount,
        galeMultiplier,
        galeLevel,
      );
      const tradeId = addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "Buscando nova entrada...",
        amount: currentAmount.toString(),
        status: "Nova oportunidade encontrada...",
        price: getCurrentPriceValue(),
      });
      setAutomation((prev) => ({
        ...prev,
        waitingForResult: false,
        lastTransactionExpirationTime: null,
      }));
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      if (automationActiveRef.current) {
        setTimeout(() => {
          if (automationActiveRef.current) {
            updateTradeInHistory(tradeId, {
              status: "Analisando..." as any,
              direction: "",
              timestamp: new Date().toLocaleTimeString("pt-BR"),
            });
            executeAutomatedTradeWithExistingId(
              true,
              galeLevel,
              baseAmount,
              tradeId,
            );
          }
        }, 10000);
      }
    }
  };

  const executeAutomatedTradeWithExistingId = async (
    forceActive = false,
    galeLevel = currentGaleLevel,
    baseAmount = entryValue,
    existingTradeId: number,
  ) => {
    if (executingTradeRef.current || isExecutingTrade) return;
    if (!forceActive && !automationActiveRef.current) return;

    const currentAmount = calculateGaleAmount(
      entryValue,
      galeMultiplier,
      galeLevel,
    );
    const currentBalanceNum = Number.parseFloat(
      automation.currentBalance.replace(/\./g, "").replace(",", "."),
    );
    if (currentBalanceNum < currentAmount) {
      automationActiveRef.current = false;
      setAutomation((prev) => ({ ...prev, isActive: false }));
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      setShowInsufficientBalanceModal(true);
      return;
    }

    executingTradeRef.current = true;
    setIsExecutingTrade(true);

    const currentProfit = automation.currentProfit;
    const stopLossValue = Number.parseFloat(stopLoss);
    const potentialProfitIfLoss = currentProfit - currentAmount;
    if (potentialProfitIfLoss <= -stopLossValue) {
      automationActiveRef.current = false;
      setAutomation((prev) => ({ ...prev, isActive: false }));
      addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "STOP LOSS",
        amount: currentAmount.toFixed(2),
        status: "Cancelada",
        result: `Stop Loss Atingido: -R$ ${stopLossValue.toFixed(2)}`,
        price: getCurrentPriceValue(),
      });
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      return;
    }

    try {
      const tradeId = existingTradeId;
      updateTradeInHistory(tradeId, {
        amount: currentAmount.toString(),
        price: getCurrentPriceValue(),
      });

      setTimeout(
        () =>
          updateTradeInHistory(tradeId, { status: "Traçando estratégia..." }),
        1000,
      );
      setTimeout(
        () => updateTradeInHistory(tradeId, { status: "Executando..." }),
        2000,
      );

      const analysis = await getAIAnalysis();
      let finalDirection = normalizeDirection(analysis.recomendacao);
      if (reverseTradesRemaining > 0) {
        const inverted = invertDirection(finalDirection);
        debugLog("Inverting direction", {
          original: finalDirection,
          inverted,
          remainingBefore: reverseTradesRemaining,
        });
        finalDirection = inverted;
        const remainingAfter = Math.max(0, reverseTradesRemaining - 1);
        setReverseTradesRemaining(remainingAfter);
        debugLog("Reverse trades remaining", remainingAfter);
        if (remainingAfter === 0) debugLog("Reverse mode finished");
      }
      updateTradeInHistory(tradeId, {
        direction: finalDirection,
        status: "Aguardando resultado...",
      });

      const currentPriceNow = getCurrentPriceValue();
      try {
        const transaction = await createTransaction(
          finalDirection,
          currentPriceNow,
          currentAmount.toString(),
        );
        updateTradeInHistory(tradeId, {
          transactionExpirationTime: new Date(
            transaction.expiration_datetime,
          ).getTime(),
        });
        scheduleTransactionResultHandling(
          transaction.transaction_id,
          transaction.expiration_datetime,
          galeLevel,
          baseAmount,
          currentAmount,
          tradeId,
        );
      } catch (transactionError) {
        console.error("Erro ao criar transação:", transactionError);
        const errorMessage =
          transactionError instanceof Error
            ? transactionError.message
            : String(transactionError);
        if (
          errorMessage.includes("400") ||
          errorMessage.includes("Failed to create transaction")
        ) {
          updateTradeInHistory(tradeId, {
            status: "Cancelada" as any,
            result: "Erro 400 - Transação cancelada",
          });
          triggerSessionExpired(
            "Erro na transação. Sua sessão foi expirada. Faça login novamente.",
          );
        } else {
          updateTradeInHistory(tradeId, {
            status: "Perdeu",
            result: "Nova oportunidade Encontrada",
          });
        }
        setAutomation((prev) => ({
          ...prev,
          waitingForResult: false,
          lastTransactionExpirationTime: null,
        }));
        executingTradeRef.current = false;
        setIsExecutingTrade(false);
        if (automationActiveRef.current) {
          setTimeout(() => {
            if (automationActiveRef.current)
              executeAutomatedTrade(true, galeLevel, baseAmount);
          }, 5000);
        }
        return;
      }
    } catch (error) {
      console.error("Erro no trade automatizado:", error);
      const currentAmount = calculateGaleAmount(
        baseAmount,
        galeMultiplier,
        galeLevel,
      );
      const tradeId = addTradeToHistory({
        timestamp: new Date().toLocaleTimeString("pt-BR"),
        direction: "Buscando nova entrada...",
        amount: currentAmount.toString(),
        status: "Nova oportunidade encontrada...",
        price: getCurrentPriceValue(),
      });
      setAutomation((prev) => ({
        ...prev,
        waitingForResult: false,
        lastTransactionExpirationTime: null,
      }));
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      if (automationActiveRef.current) {
        setTimeout(() => {
          if (automationActiveRef.current) {
            updateTradeInHistory(tradeId, {
              status: "Analisando..." as any,
              direction: "",
              timestamp: new Date().toLocaleTimeString("pt-BR"),
            });
            executeAutomatedTradeWithExistingId(
              true,
              galeLevel,
              baseAmount,
              tradeId,
            );
          }
        }, 10000);
      }
    }
  };

  const toggleAutomation = () => {
    if (automation.isActive) {
      automationActiveRef.current = false;
      setAutomation((prev) => ({ ...prev, isActive: false }));
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      setConsecutiveLosses(0);
      setReverseTradesRemaining(0);
    } else {
      setTradeHistory([]);
      if (executingTradeRef.current || isExecutingTrade) return;
      automationActiveRef.current = true;
      setAutomation((prev) => ({
        ...prev,
        isActive: true,
        currentProfit: 0,
        totalTrades: 0,
        winTrades: 0,
        lossTrades: 0,
        lastTransactionExpirationTime: null,
      }));
      setConsecutiveLosses(0);
      setReverseTradesRemaining(0);
      setCurrentGaleLevel(0);
      setCurrentEntryAmount(entryValue);
      executingTradeRef.current = false;
      setIsExecutingTrade(false);
      setTimeout(() => {
        if (automationActiveRef.current) executeAutomatedTrade(true);
      }, 500);
    }
  };

  // Callback do componente de gráfico
  const handleChartStats = (stats: LiveChartStats) => {
    setCurrentPrice(stats.currentPrice);
    setPreviousPrice(stats.previousPrice);
    setOhlc(stats.ohlc);
    setWsStatus(stats.wsStatus);
  };

  return (
    <>
      <PageTransition>
        <div className="min-h-screen bg-gradient-to-br from-gray-950 via-slate-900 to-gray-950">
          {/* Header */}
          <header className="border-b border-gray-800/50 bg-gray-900/30 backdrop-blur-xl">
            <div className="container mx-auto px-4 py-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <Button
                    variant="ghost"
                    size="icon"
                    className="text-gray-400 hover:text-white hover:bg-gray-800"
                    onClick={() => handleNavigationAttempt("/ai-operation")}
                  >
                    <ArrowLeft className="w-5 h-5" />
                  </Button>
                  <div className="flex items-center space-x-3">
                    <div className="p-2 bg-gray-800 rounded-lg">
                      <Image
                        src="/logo-optimus.png"
                        alt="OptimusIA Logo"
                        width={32}
                        height={32}
                        className="drop-shadow-lg"
                      />
                    </div>
                    <div>
                      <span className="text-xl font-bold text-white">
                        OptimusIA
                      </span>
                      <div className="flex items-center space-x-2 mt-1">
                        <Zap className="w-4 h-4 text-cyan-400" />
                        <span className="text-sm text-cyan-400 font-medium">
                          AUTOMAÇÃO AVANÇADA
                        </span>
                      </div>
                    </div>
                  </div>
                </div>
                {/* Refresh Button with cooldown */}
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-gray-400 hover:text-white hover:bg-gray-800"
                  onClick={fetchUserBalance}
                  disabled={refreshCooldown > 0}
                >
                  <RefreshCw className="w-5 h-5" />
                  {refreshCooldown > 0 && (
                    <span className="ml-1 text-xs">{refreshCooldown}s</span>
                  )}
                </Button>
              </div>
            </div>
          </header>

          {/* Main Content */}
          <main className="container mx-auto px-4 py-6">
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
              {/* Chart Section */}
              <div className="lg:col-span-2 space-y-6">
                {/* Asset Info */}
                <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                  <CardContent className="p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-3">
                        <div className="flex items-center justify-center w-12 h-12 rounded-xl bg-gray-800 overflow-hidden">
                          {imageUrl ? (
                            <img
                              src={imageUrl || "/placeholder.svg"}
                              alt={name || ""}
                              className="w-8 h-8"
                            />
                          ) : (
                            <span className="text-lg font-bold text-white">
                              {symbol?.substring(0, 2)}
                            </span>
                          )}
                        </div>
                        <div>
                          <h2 className="text-xl font-bold text-white">
                            {name}
                          </h2>
                          <p className="text-gray-400 text-sm">{symbol}</p>
                        </div>
                      </div>
                      <div className="text-right">
                        <p className="text-2xl font-bold text-white">
                          ${getCurrentPriceValue()}
                        </p>
                        <div
                          className={`flex items-center space-x-1 ${isPositive ? "text-green-400" : "text-red-400"}`}
                        >
                          {isPositive ? (
                            <TrendingUp className="w-4 h-4" />
                          ) : (
                            <TrendingDown className="w-4 h-4" />
                          )}
                          <span className="font-semibold text-sm">
                            {isPositive ? "+" : ""}
                            {change.toFixed(2)} ({isPositive ? "+" : ""}
                            {percentage.toFixed(2)}%)
                          </span>
                        </div>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                {/* Gráfico via componente (substitui todo o bloco anterior do canvas) */}
                {symbol && (
                  <LiveChart
                    ref={chartRef}
                    symbol={symbol}
                    onStatsChange={handleChartStats}
                  />
                )}
              </div>

              {/* Automation Settings */}
              <div className="lg:col-span-1 space-y-6">
                <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                  <CardContent className="p-4 space-y-4">
                    <h3 className="text-lg font-semibold text-white">
                      Configurações
                    </h3>

                    <div>
                      <Label htmlFor="entryValue" className="text-gray-300">
                        Valor da Entrada
                      </Label>
                      <Input
                        type="number"
                        id="entryValue"
                        className="bg-gray-800 border-gray-700 text-white"
                        value={entryValue}
                        onChange={(e) =>
                          setEntryValue(e.target.value.replace(",", "."))
                        }
                      />
                    </div>

                    <div>
                      <Label htmlFor="galeMultiplier" className="text-gray-300">
                        Multiplicador de Gale
                      </Label>
                      <Select
                        value={galeMultiplier}
                        onValueChange={setGaleMultiplier}
                      >
                        <SelectTrigger className="bg-gray-800 border-gray-700 text-white">
                          <SelectValue placeholder="Selecione" />
                        </SelectTrigger>
                        <SelectContent className="bg-gray-800 border-gray-700 text-white">
                          <SelectItem value="1.1">1.1 (Conservador)</SelectItem>
                          <SelectItem value="1.2">1.2 (Moderado)</SelectItem>
                          <SelectItem value="1.3">1.3 (Agressivo)</SelectItem>
                          <SelectItem value="1.4">
                            1.4 (Muito Agressivo)
                          </SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="expiration" className="text-gray-300">
                        Expiração (minutos)
                      </Label>
                      <Select value={expiration} onValueChange={setExpiration}>
                        <SelectTrigger className="bg-gray-800 border-gray-700 text-white">
                          <SelectValue placeholder="Selecione" />
                        </SelectTrigger>
                        <SelectContent className="bg-gray-800 border-gray-700 text-white">
                          <SelectItem value="1">1 minuto</SelectItem>
                          <SelectItem value="5">5 minutos</SelectItem>
                          <SelectItem value="10">10 minutos</SelectItem>
                          <SelectItem value="15">15 minutos</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="profitTarget" className="text-gray-300">
                        Meta de Lucro (R$)
                      </Label>
                      <Input
                        type="number"
                        id="profitTarget"
                        className="bg-gray-800 border-gray-700 text-white"
                        value={profitTarget}
                        onChange={(e) =>
                          setProfitTarget(e.target.value.replace(",", "."))
                        }
                      />
                    </div>

                    <div>
                      <Label htmlFor="stopLoss" className="text-gray-300">
                        Stop Loss (R$)
                      </Label>
                      <Input
                        type="number"
                        id="stopLoss"
                        className="bg-gray-800 border-gray-700 text-white"
                        value={stopLoss}
                        onChange={(e) =>
                          setStopLoss(e.target.value.replace(",", "."))
                        }
                      />
                    </div>

                    <Button
                      className="w-full bg-blue-500 hover:bg-blue-600 text-white font-semibold"
                      onClick={toggleAutomation}
                      disabled={
                        automation.waitingForResult || wsStatus !== "connected"
                      }
                    >
                      {automation.isActive ? (
                        <>
                          <Pause className="w-4 h-4 mr-2" />
                          Pausar Automação
                        </>
                      ) : (
                        <>
                          <Play className="w-4 h-4 mr-2" />
                          Iniciar Automação
                        </>
                      )}
                    </Button>
                  </CardContent>
                </Card>

                {/* Balance Info */}
                <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
                  <CardContent className="p-4 space-y-4">
                    <h3 className="text-lg font-semibold text-white">Saldo</h3>
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm text-gray-400">Saldo Atual</p>
                        <p className="text-2xl font-bold text-white">
                          R$ {automation.currentBalance}
                        </p>
                      </div>
                      <DollarSign className="w-8 h-8 text-green-400" />
                    </div>
                  </CardContent>
                </Card>
              </div>
            </div>

            {/* Total Profit Info */}
            <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl mb-6">
              <CardContent className="p-4 space-y-4">
                <h3 className="text-lg font-semibold text-white">
                  Lucro Total
                </h3>
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm text-gray-400">
                      Lucro/Prejuízo Atual
                    </p>
                    <p
                      className={`text-2xl font-bold ${automation.currentProfit >= 0 ? "text-green-400" : "text-red-400"}`}
                    >
                      {automation.currentProfit >= 0 ? "+" : ""}R${" "}
                      {automation.currentProfit.toFixed(2)}
                    </p>
                  </div>
                  <Target className="w-8 h-8 text-purple-400" />
                </div>
              </CardContent>
            </Card>

            {/* Trade History */}
            <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-xl">
              <CardContent className="p-6">
                <h3 className="text-lg font-semibold text-white mb-4">
                  Histórico de Operações
                </h3>
                <div className="space-y-3 max-h-64 overflow-y-auto">
                  {tradeHistory.length === 0 ? (
                    <p className="text-gray-400 text-center py-4">
                      Nenhuma operação realizada ainda
                    </p>
                  ) : (
                    tradeHistory.map((trade) => (
                      <div
                        key={trade.id}
                        className="flex items-center justify-between p-3 bg-gray-800/50 rounded-lg"
                      >
                        <div className="flex items-center space-x-3">
                          {getStatusIcon(trade.status)}
                          <div>
                            <div className="flex items-center space-x-2">
                              <span className="text-white font-medium">
                                {trade.direction || "..."}
                              </span>
                              <span className="text-gray-400">•</span>
                              <span className="text-gray-400">
                                R$ {trade.amount}
                              </span>
                              {(trade.status === "Executando..." ||
                                trade.status === "Aguardando resultado...") &&
                                remainingTimes[trade.id] !== undefined && (
                                  <>
                                    <span className="text-gray-400 hidden sm:inline">
                                      •
                                    </span>
                                    <span className="text-yellow-400 font-semibold text-sm md:text-base">
                                      Expiração:{" "}
                                      {Math.floor(remainingTimes[trade.id] / 60)
                                        .toString()
                                        .padStart(2, "0")}
                                      :
                                      {(remainingTimes[trade.id] % 60)
                                        .toString()
                                        .padStart(2, "0")}
                                    </span>
                                  </>
                                )}
                            </div>
                            <p className="text-xs text-gray-500">
                              {trade.timestamp}
                            </p>
                          </div>
                        </div>
                        <div className="text-right">
                          <p
                            className={`font-semibold ${getStatusColor(trade.status)}`}
                          >
                            {trade.status}
                          </p>
                          {trade.result && (
                            <p
                              className={`text-xs ${trade.result.startsWith("+") ? "text-green-400" : "text-red-400"}`}
                            >
                              {trade.result}
                            </p>
                          )}
                        </div>
                      </div>
                    ))
                  )}
                </div>
              </CardContent>
            </Card>
          </main>
        </div>

        {/* Win Notification */}
        {winNotification && (
          <div
            className={`fixed bottom-6 left-4 md:left-6 z-50 transform transition-all duration-300 ease-out ${
              winNotification.visible
                ? "translate-x-0 opacity-100 scale-100"
                : "-translate-x-full opacity-0 scale-95"
            }`}
          >
            <div className="bg-gradient-to-r from-green-500 to-emerald-500 text-white px-4 py-3 md:px-6 md:py-4 rounded-xl shadow-2xl border border-green-400/30 backdrop-blur-sm max-w-xs md:max-w-sm">
              <div className="flex items-center space-x-2 md:space-x-3">
                <div className="w-6 h-6 md:w-8 md:h-8 bg-white/20 rounded-full flex items-center justify-center flex-shrink-0">
                  <CheckCircle className="w-4 h-4 md:w-5 md:h-5 text-white" />
                </div>
                <div className="min-w-0 flex-1">
                  <p className="font-bold text-base md:text-lg truncate">
                    {winNotification.amount}
                  </p>
                  <p className="text-green-100 text-xs md:text-sm">
                    Operação vencedora!
                  </p>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Goal Notification */}
        {goalNotification && (
          <div
            className={`fixed bottom-6 left-4 md:left-6 z-50 transform transition-all duration-300 ease-out ${
              goalNotification.visible
                ? "translate-x-0 opacity-100 scale-100"
                : "-translate-x-full opacity-0 scale-95"
            }`}
          >
            <div className="bg-gradient-to-r from-purple-500 to-indigo-500 text-white px-4 py-3 md:px-6 md:py-4 rounded-xl shadow-2xl border border-purple-400/30 backdrop-blur-sm max-w-xs md:max-w-sm flex flex-col items-start">
              <div className="flex items-center space-x-2 md:space-x-3 mb-2">
                <div className="w-6 h-6 md:w-8 md:h-8 bg-white/20 rounded-full flex items-center justify-center flex-shrink-0">
                  <Target className="w-4 h-4 md:w-5 md:h-5 text-white" />
                </div>
                <div className="min-w-0 flex-1">
                  <p className="font-bold text-base md:text-lg truncate">
                    Meta Atingida: R$ {goalNotification.amount}
                  </p>
                  <p className="text-purple-100 text-xs md:text-sm">
                    Parabéns! Sua meta de lucro foi alcançada.
                  </p>
                </div>
              </div>
              <Button
                onClick={dismissGoalNotification}
                className="w-full bg-purple-700 hover:bg-purple-800 text-white text-sm py-1.5 mt-2"
              >
                Entendi
              </Button>
            </div>
          </div>
        )}

        {/* Leave Confirmation AlertDialog */}
        <AlertDialog
          open={showConfirmLeaveModal}
          onOpenChange={setShowConfirmLeaveModal}
        >
          <AlertDialogContent className="bg-gray-900 border-gray-800 text-white">
            <AlertDialogHeader>
              <AlertDialogTitle className="text-white">
                Operação em Andamento
              </AlertDialogTitle>
              <AlertDialogDescription className="text-gray-400">
                Uma operação de automação está ativa e aguardando o resultado.
                Sair agora pode interromper a operação.
                {automation.lastTransactionExpirationTime &&
                  remainingTimeForTrade > 0 && (
                    <span className="mt-2 text-lg font-bold text-blue-400 block">
                      Tempo restante: {remainingTimeForTrade} segundos
                    </span>
                  )}
                {automation.lastTransactionExpirationTime &&
                  remainingTimeForTrade <= 0 && (
                    <span className="mt-2 text-lg font-bold text-yellow-400 block">
                      Aguardando resultado final...
                    </span>
                  )}
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel
                className="bg-gray-700 hover:bg-gray-600 text-white border-gray-600"
                onClick={() => {
                  setShowConfirmLeaveModal(false);
                  setNextNavigationPath(null);
                }}
              >
                Permanecer
              </AlertDialogCancel>
              <AlertDialogAction
                className="bg-red-600 hover:bg-red-700 text-white"
                onClick={() => {
                  automationActiveRef.current = false;
                  setAutomation((prev) => ({
                    ...prev,
                    isActive: false,
                    waitingForResult: false,
                    lastTransactionExpirationTime: null,
                  }));
                  executingTradeRef.current = false;
                  setIsExecutingTrade(false);
                  if (nextNavigationPath) router.push(nextNavigationPath);
                  setShowConfirmLeaveModal(false);
                  setNextNavigationPath(null);
                }}
              >
                Sair Mesmo Assim
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        {/* Insufficient Balance AlertDialog */}
        <AlertDialog
          open={showInsufficientBalanceModal}
          onOpenChange={setShowInsufficientBalanceModal}
        >
          <AlertDialogContent className="bg-gray-900 border-gray-800 text-white">
            <AlertDialogHeader>
              <AlertDialogTitle className="text-white">
                Saldo Insuficiente
              </AlertDialogTitle>
              <AlertDialogDescription className="text-gray-400">
                Não há saldo suficiente para efetuar a próxima entrada. A
                automação foi pausada.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogAction
                className="bg-blue-600 hover:bg-blue-700 text-white"
                onClick={() => setShowInsufficientBalanceModal(false)}
              >
                Entendi
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </PageTransition>
      <SessionExpiredModal
        open={sessionExpiredOpen}
        message={sessionExpiredMessage}
        onConfirm={handleSessionExpiredRedirect}
      />
    </>
  );
}
