"use client"

import React, {
  useState,
  useEffect,
  useRef,
  useImperativeHandle,
  forwardRef,
} from "react"
import { Card, CardContent } from "@/components/ui/card"

interface ChartData {
  candle_id: number
  open_time: number
  open: string
  higher: string
  lower: string
  close: string
  volume: number
}

interface ChartResponse {
  date: string
  status: string
  symbol: string
  data: ChartData[]
}

interface TooltipData {
  x: number
  y: number
  candle: ChartData
  visible: boolean
}

type WSStatus = "connecting" | "connected" | "disconnected"

export type LiveChartStats = {
  currentPrice: string
  previousPrice: string | null
  ohlc: { open: string; high: string; low: string; close: string }
  wsStatus: WSStatus
}

export type LiveChartHandle = {
  exportAsBase64: () => Promise<string | null>
}

type Props = {
  symbol: string
  interval?: number // minutos (default 1)
  countback?: number // default 30
  onStatsChange?: (stats: LiveChartStats) => void
}

const LiveChart = forwardRef<LiveChartHandle, Props>(
  ({ symbol, interval = 1, countback = 30, onStatsChange }, ref) => {
    const [chartData, setChartData] = useState<ChartData[]>([])
    const [isLoading, setIsLoading] = useState(true)
    const [error, setError] = useState<string | null>(null)
    const [tooltip, setTooltip] = useState<TooltipData>({
      x: 0,
      y: 0,
      candle: {} as ChartData,
      visible: false,
    })
    const eventSourceRef = useRef<EventSource | null>(null)
    const [wsStatus, setWsStatus] = useState<WSStatus>("connecting")
    const canvasRef = useRef<HTMLCanvasElement>(null)
    const containerRef = useRef<HTMLDivElement>(null)

    // Expor método para a página (pegar imagem Base64 do canvas)
    useImperativeHandle(ref, () => ({
      exportAsBase64: async () => {
        const canvas = canvasRef.current
        if (!canvas) return null
        try {
          return canvas.toDataURL("image/png")
        } catch {
          return null
        }
      },
    }))

    const formatTime = (timestamp: number) => {
      const date = new Date(timestamp * 1000)
      return date.toLocaleTimeString("pt-BR", {
        hour: "2-digit",
        minute: "2-digit",
        hour12: false,
      })
    }

    const fetchInitialChartData = async (currentSymbol: string) => {
      try {
        setIsLoading(true)
        const response = await fetch(
          `/api/chart/${currentSymbol}?interval=${interval}&countback=${countback}`,
        )
        if (!response.ok) throw new Error(`Failed: ${response.status}`)
        const data: ChartResponse = await response.json()
        if (data.status === "success" && data.data) {
          setChartData(data.data)
          setError(null)
        } else {
          throw new Error("Invalid response format")
        }
      } catch (err) {
        console.error("Error fetching initial chart data:", err)
        setError("Erro ao carregar dados históricos do gráfico")
      } finally {
        setIsLoading(false)
      }
    }

    const connectWebSocket = (currentSymbol: string) => {
      if (eventSourceRef.current) {
        try { eventSourceRef.current.close() } catch {}
        eventSourceRef.current = null
      }

      setWsStatus("connecting")
      const es = new EventSource(`/api/stream/${currentSymbol}`)

      es.onopen = () => setWsStatus("connected")

      // Sanitiza e valida tick recebido do SSE
      const normalizeTick = (raw: any) => {
        const t = {
          time: Number(raw?.time),
          open: Number(raw?.open),
          high: Number(raw?.high),
          low: Number(raw?.low),
          close: Number(raw?.close),
        }
        const valid =
          isFinite(t.time) && isFinite(t.open) && isFinite(t.high) &&
          isFinite(t.low) && isFinite(t.close)
        return valid ? t : null
      }

      es.onmessage = (event) => {
        try {
          const raw = JSON.parse(event.data)
          const tick = normalizeTick(raw)
          if (!tick) return
          const intervalSec = Math.max(1, interval) * 60
          const bucketOpen = Math.floor(tick.time / intervalSec) * intervalSec
          const skewSec = 2 // tolerância para desalinhamentos sutis

          setChartData((prevData) => {
            if (prevData.length === 0) return prevData
            const updated = [...prevData]
            const last = updated[updated.length - 1]
            const lastOpen = last.open_time
            const lastCloseNum = Number(last.close)

            // Filtro de outliers extremos (ex: troca de escala repentina)
            const ratio = lastCloseNum > 0 ? Math.abs(tick.close / lastCloseNum) : 1
            if (!isFinite(ratio) || ratio > 10 || ratio < 0.1) {
              // ignora tick absurdo que quebraria a escala
              return updated
            }

            // Garantir consistência de high/low
            const baseHigh = Math.max(tick.open, tick.close)
            const baseLow = Math.min(tick.open, tick.close)
            tick.high = Math.max(tick.high, baseHigh)
            tick.low = Math.min(tick.low, baseLow)

            // FIX: manter atualização do último candle (aberto) mesmo após reconexões
            // Considera o mesmo bucket se o open_time estiver alinhado ou dentro da tolerância
            const sameBucket =
              Math.abs(bucketOpen - lastOpen) <= skewSec ||
              // fallback: buckets equivalentes quando último não estiver perfeitamente alinhado
              Math.floor(lastOpen / intervalSec) === Math.floor(bucketOpen / intervalSec)

            if (sameBucket || bucketOpen <= lastOpen) {
              // Atualiza o candle atual (ainda aberto)
              updated[updated.length - 1] = {
                ...last,
                higher: Math.max(Number(last.higher), tick.high).toString(),
                lower: Math.min(Number(last.lower), tick.low).toString(),
                close: tick.close.toString(),
                // open permanece o do bucket/candle
                volume: last.volume,
                open_time: last.open_time, // garante preservação do bucket atual
              }
              return updated
            }

            // Só cria novo candle quando de fato avançar para o próximo bucket
            if (bucketOpen >= lastOpen + intervalSec - skewSec) {
              const newCandle: ChartData = {
                candle_id: Date.now(),
                open: tick.open.toString(),
                higher: tick.high.toString(),
                lower: tick.low.toString(),
                close: tick.close.toString(),
                open_time: bucketOpen,
                volume: 0,
              }
              updated.push(newCandle)
              if (updated.length > countback) return updated.slice(-countback)
              return updated
            }

            // Qualquer outra condição (eventos atrasados) apenas mantém
            return updated
          })
        } catch (err) {
          console.error("❌ Erro ao processar dados do WebSocket:", err)
        }
      }

      es.onerror = (error) => {
        console.error("❌ Erro no WebSocket:", error)
        setWsStatus("disconnected")
        // tenta reconectar com backoff simples
        setTimeout(() => {
          if (!eventSourceRef.current || eventSourceRef.current.readyState === 2 /* CLOSED */) {
            connectWebSocket(currentSymbol)
          }
        }, 3000)
      }

      eventSourceRef.current = es
    }

    // Inicialização e cleanup
    useEffect(() => {
      if (!symbol) return
      fetchInitialChartData(symbol)
      connectWebSocket(symbol)
      return () => {
        if (eventSourceRef.current) {
          eventSourceRef.current.close()
          eventSourceRef.current = null
        }
      }
      // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [symbol])

    // Desenhar gráfico
    useEffect(() => {
      drawChart()
      if (onStatsChange) {
        const current = chartData[chartData.length - 1]
        const prev = chartData[chartData.length - 2]
        if (current) {
          onStatsChange({
            currentPrice: current.close,
            previousPrice: prev ? prev.close : null,
            ohlc: {
              open: current.open,
              high: current.higher,
              low: current.lower,
              close: current.close,
            },
            wsStatus,
          })
        }
      }
      // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [chartData, wsStatus])

    useEffect(() => {
      const handleResize = () => drawChart()
      window.addEventListener("resize", handleResize)
      return () => window.removeEventListener("resize", handleResize)
      // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [chartData])

    const drawChart = () => {
      const canvas = canvasRef.current
      if (!canvas || chartData.length === 0) return
      const ctx = canvas.getContext("2d")
      if (!ctx) return

      const container = containerRef.current
      if (container) {
        const r = container.getBoundingClientRect()
        canvas.width = r.width
        canvas.height = Math.max(300, Math.min(500, r.width * 0.6))
        canvas.style.width = `${r.width}px`
        canvas.style.height = `${canvas.height}px`
      }

      const { width, height } = canvas
      const padding = Math.max(50, width * 0.08)
      const bottomPadding = Math.max(70, height * 0.15)

      ctx.fillStyle = "#1f2937"
      ctx.fillRect(0, 0, width, height)

      const prices = chartData.flatMap((d) => [
        Number.parseFloat(d.open),
        Number.parseFloat(d.higher),
        Number.parseFloat(d.lower),
        Number.parseFloat(d.close),
      ])
      const minPrice = Math.min(...prices)
      const maxPrice = Math.max(...prices)
      const priceRange = maxPrice - minPrice
      const paddedMin = minPrice - priceRange * 0.05
      const paddedMax = maxPrice + priceRange * 0.05
      const paddedRange = paddedMax - paddedMin

      // grid
      ctx.strokeStyle = "#374151"
      ctx.lineWidth = 1
      for (let i = 0; i <= 8; i++) {
        const y = padding + (height - padding - bottomPadding) * (i / 8)
        ctx.beginPath()
        ctx.moveTo(padding, y)
        ctx.lineTo(width - padding, y)
        ctx.stroke()
      }
      const verticalLines = Math.max(6, Math.floor(chartData.length / 3))
      for (let i = 0; i <= verticalLines; i++) {
        const x = padding + (width - 2 * padding) * (i / verticalLines)
        ctx.beginPath()
        ctx.moveTo(x, padding)
        ctx.lineTo(x, height - bottomPadding)
        ctx.stroke()
      }

      const candleWidth = Math.max(3, ((width - 2 * padding) / chartData.length) * 0.7)
      const closePoints: { x: number; y: number }[] = []

      chartData.forEach((candle, index) => {
        const x = padding + (width - 2 * padding) * (index / (chartData.length - 1))
        const open = Number.parseFloat(candle.open)
        const close = Number.parseFloat(candle.close)
        const high = Number.parseFloat(candle.higher)
        const low = Number.parseFloat(candle.lower)

        const openY = padding + ((paddedMax - open) / paddedRange) * (height - padding - bottomPadding)
        const closeY = padding + ((paddedMax - close) / paddedRange) * (height - padding - bottomPadding)
        const highY = padding + ((paddedMax - high) / paddedRange) * (height - padding - bottomPadding)
        const lowY = padding + ((paddedMax - low) / paddedRange) * (height - padding - bottomPadding)

        closePoints.push({ x, y: closeY })

        const isGreen = close > open
        ctx.fillStyle = isGreen ? "#10b981" : "#ef4444"
        ctx.strokeStyle = isGreen ? "#10b981" : "#ef4444"

        ctx.lineWidth = 2
        ctx.beginPath()
        ctx.moveTo(x, highY)
        ctx.lineTo(x, lowY)
        ctx.stroke()

        const bodyTop = Math.min(openY, closeY)
        const bodyHeight = Math.abs(closeY - openY)
        ctx.fillRect(x - candleWidth / 2, bodyTop, candleWidth, Math.max(bodyHeight, 2))
      })

      if (closePoints.length > 1) {
        ctx.strokeStyle = "#3b82f6"
        ctx.lineWidth = 2
        ctx.setLineDash([5, 5])
        ctx.beginPath()
        ctx.moveTo(closePoints[0].x, closePoints[0].y)
        for (let i = 1; i < closePoints.length; i++) {
          ctx.lineTo(closePoints[i].x, closePoints[i].y)
        }
        ctx.stroke()
        ctx.setLineDash([])
      }

      // Eixos/labels
      ctx.fillStyle = "#ffffff"
      ctx.font = `${Math.max(9, Math.min(12, width * 0.012))}px Arial`
      ctx.textAlign = "right"
      ctx.strokeStyle = "#1f2937"
      ctx.lineWidth = 2
      for (let i = 0; i <= 8; i++) {
        const price = paddedMax - (paddedRange * i) / 8
        const y = padding + (height - padding - bottomPadding) * (i / 8)
        const t = price.toString()
        ctx.strokeText(t, padding - 20, y + 4)
        ctx.fillText(t, padding - 20, y + 4)
      }

      ctx.textAlign = "center"
      ctx.fillStyle = "#ffffff"
      ctx.font = `${Math.max(8, Math.min(11, width * 0.011))}px Arial`
      chartData.forEach((candle, index) => {
        const x = padding + (width - 2 * padding) * (index / (chartData.length - 1))
        if (index % Math.max(1, Math.floor(chartData.length / 5)) === 0 || index === chartData.length - 1) {
          const time = formatTime(candle.open_time)
          ctx.strokeText(time, x, height - 30)
          ctx.fillText(time, x, height - 30)
        }
      })

      if (chartData.length > 0) {
        const currentPrice = Number.parseFloat(chartData[chartData.length - 1].close)
        const currentY = padding + ((paddedMax - currentPrice) / paddedRange) * (height - padding - bottomPadding)
        ctx.strokeStyle = "#fbbf24"
        ctx.lineWidth = 2
        ctx.setLineDash([10, 5])
        ctx.beginPath()
        ctx.moveTo(padding, currentY)
        ctx.lineTo(width - padding, currentY)
        ctx.stroke()
        ctx.setLineDash([])

        ctx.fillStyle = "#fbbf24"
        ctx.font = `${Math.max(9, Math.min(12, width * 0.012))}px Arial`
        ctx.textAlign = "left"
        ctx.strokeStyle = "#1f2937"
        ctx.lineWidth = 2
        const t = currentPrice.toString()
        ctx.strokeText(t, width - padding + 15, currentY + 4)
        ctx.fillText(t, width - padding + 15, currentY + 4)
      }
    }

    const handleMouseMove = (event: React.MouseEvent<HTMLCanvasElement>) => {
      const canvas = canvasRef.current
      if (!canvas || chartData.length === 0) return
      const rect = canvas.getBoundingClientRect()
      const x = event.clientX - rect.left

      const { width } = canvas
      const padding = width < 768 ? 30 : 40
      const candleIndex = Math.round(((x - padding) / (width - 2 * padding)) * (chartData.length - 1))

      if (candleIndex >= 0 && candleIndex < chartData.length) {
        const candle = chartData[candleIndex]
        setTooltip({
          x: event.clientX,
          y: event.clientY,
          candle,
          visible: true,
        })
      }
    }

    const handleMouseLeave = () => {
      setTooltip((prev) => ({ ...prev, visible: false }))
    }

    const ohlc = (() => {
      if (chartData.length === 0) return { open: "0", high: "0", low: "0", close: "0" }
      const c = chartData[chartData.length - 1]
      return { open: c.open, high: c.higher, low: c.lower, close: c.close }
    })()

    const getWSStatusColor = () => {
      switch (wsStatus) {
        case "connected":
          return "bg-green-500"
        case "connecting":
          return "bg-yellow-500"
        case "disconnected":
          return "bg-red-500"
        default:
          return "bg-gray-500"
      }
    }

    const getWSStatusText = () => {
      switch (wsStatus) {
        case "connected":
          return "LIVE DATA"
        case "connecting":
          return "CONECTANDO"
        case "disconnected":
          return "DESCONECTADO"
        default:
          return "AGUARDANDO"
      }
    }

    return (
      <Card className="bg-gray-900 border-gray-800 shadow-lg">
        <CardContent className="p-4 md:p-6">
          <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-4 space-y-3 md:space-y-0">
            <div className="grid grid-cols-4 gap-2 md:gap-4 text-xs md:text-sm bg-gray-800/50 p-2 rounded-lg">
              <div className="text-center">
                <p className="text-gray-300 font-medium text-xs mb-1">OPEN</p>
                <p className="text-white font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                  {ohlc.open}
                </p>
              </div>
              <div className="text-center">
                <p className="text-gray-300 font-medium text-xs mb-1">HIGH</p>
                <p className="text-green-400 font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                  {ohlc.high}
                </p>
              </div>
              <div className="text-center">
                <p className="text-gray-300 font-medium text-xs mb-1">LOW</p>
                <p className="text-red-400 font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                  {ohlc.low}
                </p>
              </div>
              <div className="text-center">
                <p className="text-gray-300 font-medium text-xs mb-1">CLOSE</p>
                <p className="text-white font-semibold text-sm md:text-base bg-gray-700 px-1 py-1 rounded">
                  {ohlc.close}
                </p>
              </div>
            </div>

            <div
              className={`flex items-center space-x-2 ${wsStatus === "connected" ? "bg-green-500/20" : wsStatus === "connecting" ? "bg-yellow-500/20" : "bg-red-500/20"} px-2 py-1 rounded-lg`}
            >
              <div
                className={`w-2 h-2 rounded-full ${getWSStatusColor()} ${wsStatus === "connected" ? "animate-pulse" : ""}`}
              ></div>
              <span
                className={`text-xs md:text-sm font-medium ${wsStatus === "connected" ? "text-green-400" : wsStatus === "connecting" ? "text-yellow-400" : "text-red-400"}`}
              >
                {getWSStatusText()}
              </span>
            </div>
          </div>

          {isLoading ? (
            <div className="flex items-center justify-center h-64 md:h-96">
              <div className="text-center">
                <div className="animate-spin rounded-full h-8 w-8 md:h-12 md:w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
                <p className="text-gray-400 text-sm md:text-base">Carregando gráfico...</p>
              </div>
            </div>
          ) : error ? (
            <div className="flex items-center justify-center h-64 md:h-96">
              <div className="text-center">
                <p className="text-red-400 mb-4 text-sm md:text-base">{error}</p>
                <button
                  className="border border-gray-700 text-gray-300 hover:bg-gray-800 text-sm md:text-base bg-transparent rounded px-3 py-2"
                  onClick={() => fetchInitialChartData(symbol)}
                >
                  Tentar Novamente
                </button>
              </div>
            </div>
          ) : (
            <div className="relative w-full" ref={containerRef}>
              <canvas
                ref={canvasRef}
                className="w-full h-auto rounded-lg cursor-crosshair block"
                style={{ background: "#1f2937" }}
                onMouseMove={handleMouseMove}
                onMouseLeave={handleMouseLeave}
              />

              {tooltip.visible && (
                <div
                  className="fixed z-50 bg-gray-800 border border-gray-700 rounded-lg p-3 shadow-lg pointer-events-none"
                  style={{
                    left: tooltip.x + 10,
                    top: tooltip.y - 10,
                    transform:
                      tooltip.x > window.innerWidth - 200 ? "translateX(-100%)" : "none",
                  }}
                >
                  <div className="text-xs space-y-1">
                    <div className="flex justify-between space-x-4">
                      <span className="text-gray-400">Open:</span>
                      <span className="text-white font-semibold">{tooltip.candle.open}</span>
                    </div>
                    <div className="flex justify-between space-x-4">
                      <span className="text-gray-400">High:</span>
                      <span className="text-green-400 font-semibold">{tooltip.candle.higher}</span>
                    </div>
                    <div className="flex justify-between space-x-4">
                      <span className="text-gray-400">Low:</span>
                      <span className="text-red-400 font-semibold">{tooltip.candle.lower}</span>
                    </div>
                    <div className="flex justify-between space-x-4">
                      <span className="text-gray-400">Close:</span>
                      <span className="text-white font-semibold">{tooltip.candle.close}</span>
                    </div>
                    <div className="flex justify-between space-x-4">
                      <span className="text-gray-400">Time:</span>
                      <span className="text-blue-400 font-semibold">
                        {formatTime(tooltip.candle.open_time)}
                      </span>
                    </div>
                  </div>
                </div>
              )}
            </div>
          )}
        </CardContent>
      </Card>
    )
  },
)

LiveChart.displayName = "LiveChart"
export default LiveChart
