'use client';

import { useEffect } from 'react';
import { routeMapping } from '@/lib/route-mapping';

/**
 * Componente que intercepta TODOS os links da página
 * e converte automaticamente para rotas ofuscadas
 * 
 * Adicione este componente no layout.tsx
 */
export function LinkObfuscator() {
  useEffect(() => {
    // Criar um mapa reverso das rotas reais para ofuscadas
    const routeMap = new Map<string, string>();
    Object.entries(routeMapping).forEach(([route, hash]) => {
      routeMap.set(`/${route}`, `/${hash}`);
    });

    // Interceptar cliques em links
    const handleLinkClick = (e: Event) => {
      const target = e.target as HTMLElement;
      const link = target.closest('a') as HTMLAnchorElement | null;

      if (!link) return;

      const href = link.getAttribute('href');
      if (!href || href.startsWith('http') || href.startsWith('#')) return;

      // Verificar se a rota precisa ser ofuscada
      for (const [realRoute, obfuscatedRoute] of routeMap) {
        if (href === realRoute || href.startsWith(realRoute + '?')) {
          // Manter query strings
          const queryIndex = href.indexOf('?');
          const newHref = queryIndex > -1 
            ? obfuscatedRoute + href.substring(queryIndex)
            : obfuscatedRoute;
          
          link.setAttribute('href', newHref);
          break;
        }
      }
    };

    document.addEventListener('click', handleLinkClick, true);

    // Também ofuscar links já existentes ao montar
    const allLinks = document.querySelectorAll('a[href]');
    allLinks.forEach((link) => {
      const href = link.getAttribute('href');
      if (!href || href.startsWith('http') || href.startsWith('#')) return;

      for (const [realRoute, obfuscatedRoute] of routeMap) {
        if (href === realRoute || href.startsWith(realRoute + '?')) {
          const queryIndex = href.indexOf('?');
          const newHref = queryIndex > -1 
            ? obfuscatedRoute + href.substring(queryIndex)
            : obfuscatedRoute;
          
          link.setAttribute('href', newHref);
          break;
        }
      }
    });

    return () => {
      document.removeEventListener('click', handleLinkClick, true);
    };
  }, []);

  return null;
}
