'use client';

import { useEffect, useState } from 'react';

interface TimeDisplayProps {
  utcTime: string | undefined;
  label?: string;
  showTimezone?: boolean;
}

/**
 * Componente para exibir horário UTC convertido para local
 * Apenas para EXIBIÇÃO visual, não altera os dados
 * 
 * Uso: <TimeDisplay utcTime="14:30:45" label="Horário de Entrada" showTimezone />
 */
export function TimeDisplay({ utcTime, label = 'Horário', showTimezone = true }: TimeDisplayProps) {
  const [localTime, setLocalTime] = useState<string>('');
  const [timezone, setTimezone] = useState<string>('');

  useEffect(() => {
    if (!utcTime) return;

    try {
      // Parse do horário UTC (HH:mm:ss)
      const [hours, minutes, seconds] = utcTime.split(':').map(Number);
      
      // Criar data em UTC
      const now = new Date();
      const utcDate = new Date(
        Date.UTC(
          now.getUTCFullYear(),
          now.getUTCMonth(),
          now.getUTCDate(),
          hours,
          minutes,
          seconds || 0
        )
      );

      // Converter para local
      const localHours = String(utcDate.getHours()).padStart(2, '0');
      const localMinutes = String(utcDate.getMinutes()).padStart(2, '0');
      const localSeconds = String(utcDate.getSeconds()).padStart(2, '0');
      
      setLocalTime(`${localHours}:${localMinutes}:${localSeconds}`);

      // Obter nome da timezone
      if (showTimezone) {
        const tzName = new Intl.DateTimeFormat('pt-BR', { 
          timeZoneName: 'short' 
        }).formatToParts(utcDate).find(p => p.type === 'timeZoneName')?.value || 'UTC';
        setTimezone(tzName);
      }
    } catch (error) {
      console.error('Erro ao converter horário:', error);
      setLocalTime(utcTime);
    }
  }, [utcTime, showTimezone]);

  if (!localTime) return null;

  return (
    <div className="flex items-center gap-2">
      <span className="text-sm font-medium text-gray-600">{label}:</span>
      <span className="font-mono text-sm">
        {localTime}
        {showTimezone && timezone && (
          <span className="text-xs text-gray-500 ml-1">({timezone})</span>
        )}
      </span>
    </div>
  );
}
