import { useRouter as useNextRouter } from 'next/navigation';
import { routeMapping } from '@/lib/route-mapping';

/**
 * Hook para navegar usando rotas ofuscadas
 * Exemplo: useObfuscatedRouter().push('dashboard')
 * 
 * Garante que TODAS as navegações usem URLs ofuscadas
 */
export function useObfuscatedRouter() {
  const router = useNextRouter();

  return {
    push: (route: keyof typeof routeMapping, options?: { scroll?: boolean }) => {
      const obfuscatedRoute = routeMapping[route];
      if (!obfuscatedRoute) {
        console.warn(`Rota desconhecida: ${route}`);
        return;
      }
      // Sempre usa a rota ofuscada
      router.push(`/${obfuscatedRoute}`, options);
    },
    replace: (route: keyof typeof routeMapping) => {
      const obfuscatedRoute = routeMapping[route];
      if (!obfuscatedRoute) {
        console.warn(`Rota desconhecida: ${route}`);
        return;
      }
      // Sempre usa a rota ofuscada
      router.replace(`/${obfuscatedRoute}`);
    },
    prefetch: (route: keyof typeof routeMapping) => {
      const obfuscatedRoute = routeMapping[route];
      if (!obfuscatedRoute) {
        console.warn(`Rota desconhecida: ${route}`);
        return;
      }
      // Sempre usa a rota ofuscada
      router.prefetch(`/${obfuscatedRoute}`);
    },
  };
}

/**
 * Função para obter a rota ofuscada de uma rota real
 */
export function getObfuscatedRoute(route: keyof typeof routeMapping): string {
  return routeMapping[route] || '';
}
