"use client";

import { useCallback, useState } from "react";
import { useRouter } from "next/navigation";

const DEFAULT_MESSAGE =
  "Sua sessão expirou. Faça login novamente para continuar.";

export function useSessionExpired() {
  const router = useRouter();
  const [open, setOpen] = useState(false);
  const [message, setMessage] = useState<string>(DEFAULT_MESSAGE);

  const trigger = useCallback((customMessage?: string) => {
    if (typeof window !== "undefined") {
      // Limpa TUDO do localStorage
      localStorage.clear();
      
      // Limpa TUDO do sessionStorage
      sessionStorage.clear();
      
      // Limpa indexedDB (usado por alguns PWAs)
      if ("indexedDB" in window) {
        const databases = ["optimus", "cache", "data"];
        databases.forEach((dbName) => {
          try {
            const deleteRequest = indexedDB.deleteDatabase(dbName);
            deleteRequest.onerror = () => console.warn(`Erro ao limpar indexedDB ${dbName}`);
          } catch (e) {
            console.warn(`Não foi possível limpar ${dbName}`);
          }
        });
      }
      
      // Limpa cache API (service workers)
      if ("caches" in window) {
        caches.keys().then((cacheNames) => {
          cacheNames.forEach((cacheName) => {
            caches.delete(cacheName).catch(() => {});
          });
        });
      }
      
      // Remove cookies importantes (se houver)
      document.cookie.split(";").forEach((c) => {
        const eqPos = c.indexOf("=");
        const name = eqPos > -1 ? c.substr(0, eqPos).trim() : c.trim();
        document.cookie = name + "=;expires=Thu, 01 Jan 1970 00:00:00 GMT;path=/";
      });
    }
    setMessage(customMessage?.trim().length ? customMessage : DEFAULT_MESSAGE);
    setOpen(true);
  }, []);

  const redirect = useCallback(() => {
    setOpen(false);
    router.push("/login");
  }, [router]);

  return {
    sessionExpiredOpen: open,
    sessionExpiredMessage: message,
    triggerSessionExpired: trigger,
    handleSessionExpiredRedirect: redirect,
  };
}
