import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';
import { reverseMapping, routeMapping } from '@/lib/route-mapping';

export function middleware(request: NextRequest) {
  const pathname = request.nextUrl.pathname;
  const searchParams = request.nextUrl.search;
  
  // Verificar se o pathname é um hash ofuscado
  const realRoute = reverseMapping[pathname.slice(1)]; // Remove a barra inicial
  
  if (realRoute) {
    // Redirecionar para a rota real mantendo query strings
    return NextResponse.rewrite(
      new URL(`/${realRoute}${searchParams}`, request.url)
    );
  }
  
  // Se for uma rota real (não ofuscada), redirecionar para a versão ofuscada
  const isRealRoute = Object.keys(routeMapping).some(
    (route) => pathname === `/${route}`
  );
  
  if (isRealRoute) {
    const routeName = pathname.slice(1) as keyof typeof routeMapping;
    const obfuscatedRoute = routeMapping[routeName];
    
    if (obfuscatedRoute) {
      // Redirecionar para a rota ofuscada
      return NextResponse.redirect(
        new URL(`/${obfuscatedRoute}${searchParams}`, request.url),
        { status: 307 }
      );
    }
  }
  
  return NextResponse.next();
}

export const config = {
  matcher: [
    // Aplicar middleware a todas as rotas exceto arquivos estáticos
    '/((?!_next/static|_next/image|favicon.ico|public).*)',
  ],
};
