import type React from "react"

interface ChartData {
  candle_id: number
  open_time: number
  open: string
  higher: string
  lower: string
  close: string
  volume: number
}

interface TooltipData {
  x: number
  y: number
  candle: ChartData
  visible: boolean
}

export const drawChart = (canvas: HTMLCanvasElement | null, chartData: ChartData[]) => {
  if (!canvas || chartData.length === 0) return

  const ctx = canvas.getContext("2d")
  if (!ctx) return

  // Set canvas size to fill the container completely
  const container = canvas.parentElement
  if (container) {
    const containerRect = container.getBoundingClientRect()
    canvas.width = containerRect.width
    canvas.height = Math.max(300, Math.min(500, containerRect.width * 0.6))

    // Set CSS size to match canvas size for proper scaling
    canvas.style.width = `${containerRect.width}px`
    canvas.style.height = `${canvas.height}px`
  }

  const { width, height } = canvas
  const padding = Math.max(50, width * 0.08)
  const bottomPadding = Math.max(70, height * 0.15)

  // Clear canvas
  ctx.fillStyle = "#1f2937"
  ctx.fillRect(0, 0, width, height)

  // Calculate price range with padding
  const prices = chartData.flatMap((d) => [
    Number.parseFloat(d.open),
    Number.parseFloat(d.higher),
    Number.parseFloat(d.lower),
    Number.parseFloat(d.close),
  ])
  const minPrice = Math.min(...prices)
  const maxPrice = Math.max(...prices)
  const priceRange = maxPrice - minPrice
  const paddedMin = minPrice - priceRange * 0.05
  const paddedMax = maxPrice + priceRange * 0.05
  const paddedRange = paddedMax - paddedMin

  // Draw grid lines
  ctx.strokeStyle = "#374151"
  ctx.lineWidth = 1

  // Horizontal grid lines
  for (let i = 0; i <= 8; i++) {
    const y = padding + (height - padding - bottomPadding) * (i / 8)
    ctx.beginPath()
    ctx.moveTo(padding, y)
    ctx.lineTo(width - padding, y)
    ctx.stroke()
  }

  // Vertical grid lines
  const verticalLines = Math.max(6, Math.floor(chartData.length / 3))
  for (let i = 0; i <= verticalLines; i++) {
    const x = padding + (width - 2 * padding) * (i / verticalLines)
    ctx.beginPath()
    ctx.moveTo(x, padding)
    ctx.lineTo(x, height - bottomPadding)
    ctx.stroke()
  }

  // Draw candlesticks
  const candleWidth = Math.max(3, ((width - 2 * padding) / chartData.length) * 0.7)
  const closePoints: { x: number; y: number }[] = []

  chartData.forEach((candle, index) => {
    const x = padding + (width - 2 * padding) * (index / (chartData.length - 1))

    const open = Number.parseFloat(candle.open)
    const close = Number.parseFloat(candle.close)
    const high = Number.parseFloat(candle.higher)
    const low = Number.parseFloat(candle.lower)

    const openY = padding + ((paddedMax - open) / paddedRange) * (height - padding - bottomPadding)
    const closeY = padding + ((paddedMax - close) / paddedRange) * (height - padding - bottomPadding)
    const highY = padding + ((paddedMax - high) / paddedRange) * (height - padding - bottomPadding)
    const lowY = padding + ((paddedMax - low) / paddedRange) * (height - padding - bottomPadding)

    // Store close points for trend line
    closePoints.push({ x, y: closeY })

    // Determine candle color
    const isGreen = close > open
    ctx.fillStyle = isGreen ? "#10b981" : "#ef4444"
    ctx.strokeStyle = isGreen ? "#10b981" : "#ef4444"

    // Draw wick
    ctx.lineWidth = 2
    ctx.beginPath()
    ctx.moveTo(x, highY)
    ctx.lineTo(x, lowY)
    ctx.stroke()

    // Draw body
    const bodyTop = Math.min(openY, closeY)
    const bodyHeight = Math.abs(closeY - openY)
    ctx.fillRect(x - candleWidth / 2, bodyTop, candleWidth, Math.max(bodyHeight, 2))
  })

  // Draw trend line connecting close prices
  if (closePoints.length > 1) {
    ctx.strokeStyle = "#3b82f6"
    ctx.lineWidth = 2
    ctx.setLineDash([5, 5])
    ctx.beginPath()
    ctx.moveTo(closePoints[0].x, closePoints[0].y)
    for (let i = 1; i < closePoints.length; i++) {
      ctx.lineTo(closePoints[i].x, closePoints[i].y)
    }
    ctx.stroke()
    ctx.setLineDash([])
  }

  // Draw price labels with better visibility
  ctx.fillStyle = "#ffffff"
  ctx.font = `${Math.max(9, Math.min(12, width * 0.012))}px Arial`
  ctx.textAlign = "right"
  ctx.strokeStyle = "#1f2937"
  ctx.lineWidth = 2

  for (let i = 0; i <= 8; i++) {
    const price = paddedMax - (paddedRange * i) / 8
    const y = padding + (height - padding - bottomPadding) * (i / 8)
    const priceText = price.toFixed(2) // Format to 2 decimal places

    // Draw text outline for better visibility
    ctx.strokeText(priceText, padding - 20, y + 4)
    ctx.fillText(priceText, padding - 20, y + 4)
  }

  // Draw time labels with better visibility
  ctx.textAlign = "center"
  ctx.fillStyle = "#ffffff"
  ctx.font = `${Math.max(8, Math.min(11, width * 0.011))}px Arial`

  // Draw time labels for each candle position (moving with the chart)
  chartData.forEach((candle, index) => {
    const x = padding + (width - 2 * padding) * (index / (chartData.length - 1))

    // Only show time labels for every few candles to avoid overcrowding
    if (index % Math.max(1, Math.floor(chartData.length / 5)) === 0 || index === chartData.length - 1) {
      const time = formatTime(candle.open_time)

      // Draw text outline for better visibility
      ctx.strokeText(time, x, height - 30)
      ctx.fillText(time, x, height - 30)
    }
  })

  // Draw current price indicator
  if (chartData.length > 0) {
    const currentPrice = Number.parseFloat(chartData[chartData.length - 1].close)
    const currentY = padding + ((paddedMax - currentPrice) / paddedRange) * (height - padding - bottomPadding)

    // Draw horizontal line for current price
    ctx.strokeStyle = "#fbbf24"
    ctx.lineWidth = 2
    ctx.setLineDash([10, 5])
    ctx.beginPath()
    ctx.moveTo(padding, currentY)
    ctx.lineTo(width - padding, currentY)
    ctx.stroke()
    ctx.setLineDash([])

    // Draw current price label
    ctx.fillStyle = "#fbbf24"
    ctx.font = `${Math.max(9, Math.min(12, width * 0.012))}px Arial`
    ctx.textAlign = "left"
    ctx.strokeStyle = "#1f2937"
    ctx.lineWidth = 2
    const currentPriceText = currentPrice.toFixed(2) // Format to 2 decimal places
    ctx.strokeText(currentPriceText, width - padding + 15, currentY + 4)
    ctx.fillText(currentPriceText, width - padding + 15, currentY + 4)
  }
}

export const handleMouseMove = (
  event: React.MouseEvent<HTMLCanvasElement>,
  chartData: ChartData[],
  setTooltip: React.Dispatch<React.SetStateAction<TooltipData>>
) => {
  const canvas = event.currentTarget
  if (!canvas || chartData.length === 0) return

  const rect = canvas.getBoundingClientRect()
  const x = event.clientX - rect.left
  const y = event.clientY - rect.top

  const { width } = canvas
  const padding = Math.max(50, width * 0.08)

  // Find the closest candle
  const candleIndex = Math.round(((x - padding) / (width - 2 * padding)) * (chartData.length - 1))

  if (candleIndex >= 0 && candleIndex < chartData.length) {
    const candle = chartData[candleIndex]
    setTooltip({
      x: event.clientX,
      y: event.clientY,
      candle,
      visible: true,
    })
  }
}

export const handleMouseLeave = (setTooltip: React.Dispatch<React.SetStateAction<TooltipData>>) => {
  setTooltip((prev) => ({ ...prev, visible: false }))
}

const formatTime = (timestamp: number) => {
  const date = new Date(timestamp * 1000)
  return date.toLocaleTimeString("pt-BR", {
    hour: "2-digit",
    minute: "2-digit",
    hour12: false,
  })
}
