"use client"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Loader2, ArrowLeft, Bot, ChevronDown, Search } from "lucide-react"
import Image from "next/image"
import { AssetItem } from "@/components/asset-item"
import { PageTransition } from "@/components/page-transition"

interface Symbol {
  id: number
  code: string
  type: string
  name: string
  status: number
  is_otc: number
  is_market_open: number
  img: string
  last_price: string
  first_price_day: string
  open_hour: string
  close_hour: string
  blocked_weekend: number
  daily_percent_variation: number
  payout: number
}

export default function AIOperationPage() {
  const [symbols, setSymbols] = useState<Symbol[]>([])
  const [filteredSymbols, setFilteredSymbols] = useState<Symbol[]>([])
  const [selectedSymbol, setSelectedSymbol] = useState<string>("")
  const [selectedSymbolName, setSelectedSymbolName] = useState<string>("")
  const [selectedModel, setSelectedModel] = useState<string>("chatgpt")
  const [assetType, setAssetType] = useState<string>("todos")
  const [searchQuery, setSearchQuery] = useState<string>("")
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [isDropdownOpen, setIsDropdownOpen] = useState(false)
  const router = useRouter()
  const [selectedSymbolType, setSelectedSymbolType] = useState<string>("")
  const [selectedSymbolImg, setSelectedSymbolImg] = useState<string>("")

  useEffect(() => {
    const fetchSymbols = async () => {
      try {
        setIsLoading(true)
        const response = await fetch("/api/symbols", { cache: "no-store" })

        if (!response.ok) {
          throw new Error(`Failed to fetch symbols: ${response.status}`)
        }

        const data = await response.json()
        console.log("API Response:", data)

        // Handle different response structures
        let symbolsArray: Symbol[] = []

        if (Array.isArray(data)) {
          symbolsArray = data
        } else if (data && typeof data === "object" && data.symbols && Array.isArray(data.symbols)) {
          symbolsArray = data.symbols
        } else if (data && typeof data === "object") {
          // If data is an object, try to find an array property
          const possibleArrays = Object.values(data).filter(Array.isArray)
          if (possibleArrays.length > 0) {
            symbolsArray = possibleArrays[0] as Symbol[]
          }
        }

        console.log("Processed symbols array:", symbolsArray)

        if (!Array.isArray(symbolsArray) || symbolsArray.length === 0) {
          throw new Error("Could not extract symbols array from response")
        }

        setSymbols(symbolsArray)
        setFilteredSymbols(symbolsArray)

        // Set first symbol as default if available
        if (symbolsArray.length > 0) {
          setSelectedSymbol(symbolsArray[0].code)
          setSelectedSymbolName(symbolsArray[0].name)
          setSelectedSymbolType(symbolsArray[0].type)
          setSelectedSymbolImg(symbolsArray[0].img)
        }
      } catch (error) {
        console.error("Error fetching symbols:", error)
        setError("Failed to load assets. Please try again.")
      } finally {
        setIsLoading(false)
      }
    }

    fetchSymbols()
  }, [])

  useEffect(() => {
    // Filter symbols based on search query and asset type
    let filtered = [...symbols]

    if (searchQuery) {
      filtered = filtered.filter(
        (symbol) =>
          symbol.code.toLowerCase().includes(searchQuery.toLowerCase()) ||
          symbol.name.toLowerCase().includes(searchQuery.toLowerCase()),
      )
    }

    if (assetType !== "todos") {
      // Filter based on the type field from the API
      if (assetType === "moedas") {
        filtered = filtered.filter((symbol) => symbol.type === "forex")
      } else if (assetType === "acoes") {
        filtered = filtered.filter((symbol) => symbol.type === "stocks")
      } else if (assetType === "cripto") {
        filtered = filtered.filter((symbol) => symbol.type === "crypto")
      }
    }

    setFilteredSymbols(filtered)
  }, [searchQuery, assetType, symbols])

  const handleSymbolSelect = (symbol: Symbol) => {
    setSelectedSymbol(symbol.code)
    setSelectedSymbolName(symbol.name)
    setSelectedSymbolType(symbol.type)
    setSelectedSymbolImg(symbol.img)
    setIsDropdownOpen(false)
  }

  const formatAssetType = (type: string) => {
    if (type === "crypto") return "Cripto"
    if (type === "forex") return "Moedas"
    if (type === "stocks") return "Ações"
    return type
  }

  const handleStartAnalysis = () => {
    if (!selectedSymbol) return

    // Store parameters in localStorage instead of URLSearchParams
    localStorage.setItem(
      "aiOperationParams",
      JSON.stringify({
        symbol: selectedSymbol,
        name: selectedSymbolName,
        img: selectedSymbolImg,
        model: selectedModel,
      }),
    )

    router.push(`/analysis`) // Navigate without query parameters
  }

  return (
    <PageTransition>
      <div className="min-h-screen bg-gray-950">
        {/* Header */}
        <header className="border-b border-gray-800 bg-gray-900">
          <div className="container mx-auto px-4 py-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-gray-400 hover:text-white hover:bg-gray-800"
                  onClick={() => router.push("/dashboard")}
                >
                  <ArrowLeft className="w-5 h-5" />
                </Button>
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-gray-800 rounded-lg">
                    <Image
                      src="/logo-optimus.png"
                      alt="OptimusIA Logo"
                      width={32}
                      height={32}
                      className="drop-shadow-lg"
                    />
                  </div>
                  <span className="text-xl font-bold text-white">OptimusIA</span>
                </div>
              </div>
            </div>
          </div>
        </header>

        {/* Main Content */}
        <main className="container mx-auto px-4 py-8">
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-white mb-2">Operação com IA</h1>
            <p className="text-gray-400">Selecione o modelo de IA e o ativo para começar</p>
          </div>

          {/* AI Model Selection */}
          <Card className="bg-gray-900 border-gray-800 shadow-lg mb-8">
            <CardContent className="p-6">
              <div className="mb-6">
                <h2 className="text-xl font-semibold text-white mb-4">Selecione o Modelo de IA</h2>
                <div className="grid grid-cols-4 gap-0">
                  <button
                    onClick={() => setSelectedModel("chatgpt")}
                    className={`flex flex-col items-center justify-center py-6 px-4 rounded-none transition-colors ${
                      selectedModel === "chatgpt" ? "bg-blue-600" : "bg-gray-800 hover:bg-gray-700"
                    }`}
                  >
                    <div className="w-10 h-10 bg-green-500 rounded-md flex items-center justify-center mb-2">
                      <svg viewBox="0 0 24 24" width="24" height="24" fill="white">
                        <path d="M22.2819 9.8211a5.9847 5.9847 0 0 0-.5157-4.9108 6.0462 6.0462 0 0 0-6.5098-2.9A6.0651 6.0651 0 0 0 4.9807 4.1818a5.9847 5.9847 0 0 0-3.9977 2.9 6.0462 6.0462 0 0 0 .7427 7.0966 5.98 5.98 0 0 0 .511 4.9107 6.051 6.051 0 0 0 6.5146 2.9001A5.9847 5.9847 0 0 0 13.2599 24a6.0557 6.0557 0 0 0 5.7718-4.2058 5.9894 5.9894 0 0 0 3.9977-2.9001 6.0557 6.0557 0 0 0-.7475-7.0729zm-9.022 12.6081a4.4755 4.4755 0 0 1-2.8764-1.0408l.1419-.0804 4.7783-2.7582a.7948.7948 0 0 0 .3927-.6813v-6.7369l2.02 1.1686a.071.071 0 0 1 .038.052v5.5826a4.504 4.504 0 0 1-4.4945 4.4944zm-9.6607-4.1254a4.4708 4.4708 0 0 1-.5346-3.0137l.142.0852 4.783 2.7582a.7712.7712 0 0 0 .7806 0l5.8428-3.3685v2.3324a.0804.0804 0 0 1-.0332.0615L9.74 19.9502a4.4992 4.4992 0 0 1-6.1408-1.6464zM2.3408 7.8956a4.485 4.485 0 0 1 2.3655-1.9728V11.6a.7664.7664 0 0 0 .3879.6765l5.8144 3.3543-2.0201 1.1685a.0757.0757 0 0 1-.071 0l-4.8303-2.7865A4.504 4.504 0 0 1 2.3408 7.872zm16.5963 3.8558L13.1038 8.364 15.1192 7.2a.0757.0757 0 0 1 .071 0l4.8303 2.7913a4.4944 4.4944 0 0 1-.6765 8.1042v-5.6772a.79.79 0 0 0-.407-.667zm2.0107-3.0231l-.142-.0852-4.7735-2.7818a.7759.7759 0 0 0-.7854 0L9.409 9.2297V6.8974a.0662.0662 0 0 1 .0284-.0615l4.8303-2.7866a4.4992 4.4992 0 0 1 6.6802 4.66zM8.3065 12.863l-2.02-1.1638a.0804.0804 0 0 1-.038-.0567V6.0742a4.4992 4.4992 0 0 1 7.3757-3.4537l-.142.0805L8.704 5.459a.7948.7948 0 0 0-.3927.6813zm1.0976-2.3654l2.602-1.4998 2.6069 1.4998v2.9994l-2.5974 1.4997-2.6067-1.4997Z" />
                      </svg>
                    </div>
                    <span className="text-sm text-white">Chat GPT</span>
                  </button>
                  <button
                    onClick={() => setSelectedModel("gpt4")}
                    className={`flex flex-col items-center justify-center py-6 px-4 rounded-none transition-colors ${
                      selectedModel === "gpt4" ? "bg-blue-600" : "bg-gray-800 hover:bg-gray-700"
                    }`}
                  >
                    <div className="w-10 h-10 bg-black rounded-md flex items-center justify-center mb-2">
                      <svg viewBox="0 0 24 24" width="24" height="24" fill="white">
                        <path d="M22.2819 9.8211a5.9847 5.9847 0 0 0-.5157-4.9108 6.0462 6.0462 0 0 0-6.5098-2.9A6.0651 6.0651 0 0 0 4.9807 4.1818a5.9847 5.9847 0 0 0-3.9977 2.9 6.0462 6.0462 0 0 0 .7427 7.0966 5.98 5.98 0 0 0 .511 4.9107 6.051 6.051 0 0 0 6.5146 2.9001A5.9847 5.9847 0 0 0 13.2599 24a6.0557 6.0557 0 0 0 5.7718-4.2058 5.9894 5.9894 0 0 0 3.9977-2.9001 6.0557 6.0557 0 0 0-.7475-7.0729zm-9.022 12.6081a4.4755 4.4755 0 0 1-2.8764-1.0408l.1419-.0804 4.7783-2.7582a.7948.7948 0 0 0 .3927-.6813v-6.7369l2.02 1.1686a.071.071 0 0 1 .038.052v5.5826a4.504 4.504 0 0 1-4.4945 4.4944zm-9.6607-4.1254a4.4708 4.4708 0 0 1-.5346-3.0137l.142.0852 4.783 2.7582a.7712.7712 0 0 0 .7806 0l5.8428-3.3685v2.3324a.0804.0804 0 0 1-.0332.0615L9.74 19.9502a4.4992 4.4992 0 0 1-6.1408-1.6464zM2.3408 7.8956a4.485 4.485 0 0 1 2.3655-1.9728V11.6a.7664.7664 0 0 0 .3879.6765l5.8144 3.3543-2.0201 1.1685a.0757.0757 0 0 1-.071 0l-4.8303-2.7865A4.504 4.504 0 0 1 2.3408 7.872zm16.5963 3.8558L13.1038 8.364 15.1192 7.2a.0757.0757 0 0 1 .071 0l4.8303 2.7913a4.4944 4.4944 0 0 1-.6765 8.1042v-5.6772a.79.79 0 0 0-.407-.667zm2.0107-3.0231l-.142-.0852-4.7735-2.7818a.7759.7759 0 0 0-.7854 0L9.409 9.2297V6.8974a.0662.0662 0 0 1 .0284-.0615l4.8303-2.7866a4.4992 4.4992 0 0 1 6.6802 4.66zM8.3065 12.863l-2.02-1.1638a.0804.0804 0 0 1-.038-.0567V6.0742a4.4992 4.4992 0 0 1 7.3757-3.4537l-.142.0805L8.704 5.459a.7948.7948 0 0 0-.3927.6813zm1.0976-2.3654l2.602-1.4998 2.6069 1.4998v2.9994l-2.5974 1.4997-2.6067-1.4997Z" />
                      </svg>
                    </div>
                    <span className="text-sm text-white">GPT-4</span>
                  </button>
                  <button
                    onClick={() => setSelectedModel("gemini")}
                    className={`flex flex-col items-center justify-center py-6 px-4 rounded-none transition-colors ${
                      selectedModel === "gemini" ? "bg-blue-600" : "bg-gray-800 hover:bg-gray-700"
                    }`}
                  >
                    <div className="w-10 h-10 bg-gray-900 rounded-md flex items-center justify-center mb-2">
                      <svg width="24" height="24" viewBox="0 0 192 192" fill="none">
                        <path d="M96 69.8L122.2 96L96 122.2L69.8 96L96 69.8Z" fill="white" />
                        <path d="M96 29L142.4 75.4L165.9 51.9L96 -18L26.1 51.9L49.6 75.4L96 29Z" fill="white" />
                        <path d="M96 163L49.6 116.6L26.1 140.1L96 210L165.9 140.1L142.4 116.6L96 163Z" fill="white" />
                        <path d="M26.1 51.9L0 78L26.1 104.1L52.2 78L26.1 51.9Z" fill="white" />
                        <path d="M165.9 51.9L139.8 78L165.9 104.1L192 78L165.9 51.9Z" fill="white" />
                      </svg>
                    </div>
                    <span className="text-sm text-white">Gemini</span>
                  </button>
                  <button
                    onClick={() => setSelectedModel("grok3")}
                    className={`flex flex-col items-center justify-center py-6 px-4 rounded-none transition-colors ${
                      selectedModel === "grok3" ? "bg-blue-600" : "bg-gray-800 hover:bg-gray-700"
                    }`}
                  >
                    <div className="w-10 h-10 bg-white rounded-md flex items-center justify-center mb-2">
                      <svg width="24" height="24" viewBox="0 0 24 24" fill="none">
                        <path d="M12 2L2 7L12 12L22 7L12 2Z" fill="black" />
                        <path d="M2 17L12 22L22 17V7L12 12L2 7V17Z" fill="black" />
                      </svg>
                    </div>
                    <span className="text-sm text-white">Grok 3</span>
                  </button>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Asset Selection */}
          <Card className="bg-gray-900 border-gray-800 shadow-lg mb-8">
            <CardContent className="p-6">
              <div>
                <h2 className="text-xl font-semibold text-white mb-4">Selecione o Ativo</h2>

                {/* Asset Dropdown Trigger */}
                <div
                  className="flex items-center justify-between p-3 bg-blue-500 rounded-lg cursor-pointer mb-4"
                  onClick={() => setIsDropdownOpen(!isDropdownOpen)}
                >
                  <div className="flex items-center space-x-2">
                    {selectedSymbol ? (
                      <>
                        <div className="flex items-center justify-center w-8 h-8 rounded-md bg-blue-600/50 overflow-hidden">
                          {selectedSymbolImg ? (
                            <img
                              src={selectedSymbolImg || "/placeholder.svg"}
                              alt={selectedSymbolName}
                              className="w-6 h-6"
                            />
                          ) : (
                            <span className="text-sm font-bold text-white">{selectedSymbol.substring(0, 2)}</span>
                          )}
                        </div>
                        <div className="flex flex-col">
                          <span className="text-white font-medium">{selectedSymbolName}</span>
                          <span className="text-blue-200 text-xs">{formatAssetType(selectedSymbolType)}</span>
                        </div>
                      </>
                    ) : (
                      <span className="text-white">Selecione um ativo</span>
                    )}
                  </div>
                  <ChevronDown
                    className={`w-5 h-5 text-white transition-transform ${isDropdownOpen ? "transform rotate-180" : ""}`}
                  />
                </div>

                {/* Asset Selection Panel */}
                {isDropdownOpen && (
                  <div className="bg-gray-900 border border-gray-800 rounded-lg shadow-lg mb-4">
                    {/* Search Bar */}
                    <div className="p-4 border-b border-gray-800">
                      <div className="relative">
                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 w-4 h-4" />
                        <Input
                          type="text"
                          placeholder="Pesquisar ativos..."
                          className="pl-10 bg-gray-800 border-gray-700 text-white"
                          value={searchQuery}
                          onChange={(e) => setSearchQuery(e.target.value)}
                        />
                      </div>
                    </div>

                    {/* Asset Type Tabs */}
                    <div className="grid grid-cols-4 gap-1 p-2 border-b border-gray-800">
                      <Button
                        variant={assetType === "todos" ? "default" : "ghost"}
                        className={assetType === "todos" ? "bg-blue-600" : "text-gray-400"}
                        onClick={() => setAssetType("todos")}
                      >
                        Todos
                      </Button>
                      <Button
                        variant={assetType === "moedas" ? "default" : "ghost"}
                        className={assetType === "moedas" ? "bg-blue-600" : "text-gray-400"}
                        onClick={() => setAssetType("moedas")}
                      >
                        Moedas
                      </Button>
                      <Button
                        variant={assetType === "acoes" ? "default" : "ghost"}
                        className={assetType === "acoes" ? "bg-blue-600" : "text-gray-400"}
                        onClick={() => setAssetType("acoes")}
                      >
                        Ações
                      </Button>
                      <Button
                        variant={assetType === "cripto" ? "default" : "ghost"}
                        className={assetType === "cripto" ? "bg-blue-600" : "text-gray-400"}
                        onClick={() => setAssetType("cripto")}
                      >
                        Cripto
                      </Button>
                    </div>

                    {/* Asset List */}
                    <div className="max-h-80 overflow-y-auto p-2 space-y-2">
                      {isLoading ? (
                        <div className="flex items-center justify-center py-8">
                          <Loader2 className="h-8 w-8 animate-spin text-blue-500" />
                          <span className="ml-2 text-gray-400">Carregando ativos...</span>
                        </div>
                      ) : error ? (
                        <div className="text-red-400 py-4 text-center">
                          {error}
                          <Button
                            variant="outline"
                            className="mt-2 border-gray-700 hover:bg-gray-800 bg-transparent"
                            onClick={() => window.location.reload()}
                          >
                            Tentar novamente
                          </Button>
                        </div>
                      ) : filteredSymbols.length === 0 ? (
                        <div className="text-gray-400 py-4 text-center">Nenhum ativo encontrado</div>
                      ) : (
                        filteredSymbols.map((symbol) => (
                          <AssetItem
                            key={symbol.id}
                            symbol={symbol.code || ""}
                            name={symbol.name || ""}
                            type={symbol.type || ""}
                            imageUrl={symbol.img || ""}
                            isActive={symbol.status === 1}
                            isSelected={selectedSymbol === symbol.code}
                            onClick={() => handleSymbolSelect(symbol)}
                          />
                        ))
                      )}
                    </div>
                  </div>
                )}

                {/* Start Analysis Button */}
                <Button
                  className="w-full bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 text-white font-semibold py-3 shadow-lg hover:shadow-blue-500/25 transition-all duration-300"
                  disabled={!selectedSymbol || isLoading}
                  onClick={handleStartAnalysis}
                >
                  <Bot className="mr-2 h-5 w-5" />
                  Iniciar Análise com {selectedModel.toUpperCase()}
                </Button>
              </div>
            </CardContent>
          </Card>
        </main>
      </div>
    </PageTransition>
  )
}
