"use client"

import { useState, useEffect } from "react"
import { useRouter, useSearchParams } from "next/navigation"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { ArrowLeft, TrendingUp, TrendingDown, Clock, Target, Brain } from "lucide-react"
import Image from "next/image"
import { PageTransition } from "@/components/page-transition"

interface AnalysisResult {
  recomendacao: string
  probabilidade: string
  explicacao: string
  entrada: string // esperado como "HH:mm:ss" em UTC
}

export default function AIResultPage() {
  const router = useRouter()
  const searchParams = useSearchParams()

  // Estados para os dados escolhidos anteriormente
  const [symbol, setSymbol] = useState<string>("")
  const [name, setName] = useState<string>("")
  const [imageUrl, setImageUrl] = useState<string>("")
  const [model, setModel] = useState<string>("")

  const [analysisResult, setAnalysisResult] = useState<AnalysisResult | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // 1) Tenta pegar da URL (fallback)
    const urlSymbol = searchParams.get("symbol") || ""
    const urlName = searchParams.get("name") || ""
    const urlImg = searchParams.get("img") || ""
    const urlModel = searchParams.get("model") || ""

    if (urlSymbol || urlName || urlImg || urlModel) {
      setSymbol(urlSymbol)
      setName(urlName)
      setImageUrl(urlImg)
      setModel(urlModel)
    } else {
      // 2) Caso comum: ler do localStorage (foi salvo em /ai-operation)
      const stored = localStorage.getItem("aiOperationParams")
      if (stored) {
        try {
          const { symbol, name, img, model } = JSON.parse(stored)
          setSymbol(symbol || "")
          setName(name || "")
          setImageUrl(img || "")
          setModel(model || "")
        } catch {
          // ignore parse error
        }
      }
    }

    // Resultado da análise (já estava usando localStorage)
    const result = localStorage.getItem("aiAnalysisResult")
    if (result) {
      setAnalysisResult(JSON.parse(result))
      localStorage.removeItem("aiAnalysisResult") // Clean up
    }

    setIsLoading(false)
  }, [searchParams])

  // === AJUSTE DE FUSO/HORÁRIO LOCAL (ÚNICA MUDANÇA FUNCIONAL) ===
  // Converte "HH:mm:ss" (interpretado como UTC) para o horário local do usuário (pt-BR)
  const formatEntradaToLocal = (entrada?: string) => {
    if (!entrada) return ""
    // Aceita "HH:mm" ou "HH:mm:ss"
    const [hhStr, mmStr, ssStr = "00"] = entrada.split(":")
    const hh = Number(hhStr ?? 0)
    const mm = Number(mmStr ?? 0)
    const ss = Number(ssStr ?? 0)

    // Cria um Date em UTC no dia de hoje
    const now = new Date()
    const utcDate = new Date(Date.UTC(now.getFullYear(), now.getMonth(), now.getDate(), hh, mm, ss))

    // Renderiza no horário local do navegador/servidor
    return utcDate.toLocaleTimeString("pt-BR", {
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit",
      hour12: false,
    })
  }
  // =============================================================

  const isCompra = analysisResult?.recomendacao?.toLowerCase().includes("compra")

  if (isLoading) {
    return (
      <PageTransition>
        <div className="min-h-screen bg-gray-950 flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500 mx-auto mb-4"></div>
            <p className="text-white text-lg">Carregando análise...</p>
          </div>
        </div>
      </PageTransition>
    )
  }

  if (!analysisResult) {
    return (
      <PageTransition>
        <div className="min-h-screen bg-gray-950 flex items-center justify-center">
          <div className="text-center">
            <p className="text-red-400 text-lg mb-4">Nenhuma análise encontrada</p>
            <Button onClick={() => router.push("/analysis")} className="bg-blue-600 hover:bg-blue-700">
              Voltar para Análise
            </Button>
          </div>
        </div>
      </PageTransition>
    )
  }

  // Usa a hora convertida para exibir ao usuário
  const entradaLocal = formatEntradaToLocal(analysisResult.entrada)

  return (
    <PageTransition>
      <div className="min-h-screen bg-gray-950">
        {/* Header */}
        <header className="border-b border-gray-800 bg-gray-900">
          <div className="container mx-auto px-4 py-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-gray-400 hover:text-white hover:bg-gray-800"
                  onClick={() => router.back()}
                >
                  <ArrowLeft className="w-5 h-5" />
                </Button>
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-gray-800 rounded-lg">
                    <Image
                      src="/logo-optimus.png"
                      alt="OptimusIA Logo"
                      width={32}
                      height={32}
                      className="drop-shadow-lg"
                    />
                  </div>
                  <span className="text-xl font-bold text-white">OptimusIA</span>
                </div>
              </div>
            </div>
          </div>
        </header>

        {/* Main Content */}
        <main className="container mx-auto px-4 py-4 md:py-8">
          {/* Asset Header */}
          <div className="mb-6 md:mb-8">
            <div className="flex items-center space-x-3 md:space-x-4 mb-4">
              <div className="flex items-center justify-center w-12 h-12 md:w-16 md:h-16 rounded-xl bg-gray-800 overflow-hidden">
                {imageUrl ? (
                  <img src={imageUrl || "/placeholder.svg"} alt={name} className="w-8 h-8 md:w-12 md:h-12" />
                ) : (
                  <span className="text-lg md:text-xl font-bold text-white">{symbol.substring(0, 2)}</span>
                )}
              </div>
              <div>
                <h1 className="text-2xl md:3xl font-bold text-white">{name}</h1>
                <p className="text-gray-400 text-sm md:text-base">{symbol}</p>
              </div>
            </div>

            <div className="flex items-center space-x-2 mb-4">
              <Brain className="w-5 h-5 text-blue-400" />
              <span className="text-blue-400 font-semibold">
                Análise com {model ? model.toUpperCase() : "—"}
              </span>
            </div>
          </div>

          {/* Analysis Result */}
          <div className="grid gap-6 md:gap-8">
            {/* Recommendation Card */}
            <Card
              className={`border-2 shadow-xl ${
                isCompra
                  ? "bg-gradient-to-br from-green-900/50 to-green-800/30 border-green-500/50"
                  : "bg-gradient-to-br from-red-900/50 to-red-800/30 border-red-500/50"
              }`}
            >
              <CardContent className="p-6 md:p-8">
                <div className="flex items-center justify-between mb-6">
                  <div className="flex items-center space-x-3">
                    {isCompra ? (
                      <TrendingUp className="w-8 h-8 text-green-400" />
                    ) : (
                      <TrendingDown className="w-8 h-8 text-red-400" />
                    )}
                    <div>
                      <h2 className="text-2xl md:text-3xl font-bold text-white">
                        {analysisResult.recomendacao.toUpperCase()}
                      </h2>
                      <p className="text-gray-300">Recomendação da IA</p>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className={`text-3xl md:text-4xl font-bold ${isCompra ? "text-green-400" : "text-red-400"}`}>
                      {analysisResult.probabilidade}
                    </div>
                    <p className="text-gray-300 text-sm">Probabilidade</p>
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                    <div className="flex items-center space-x-2">
                      <Target className="w-5 h-5 text-blue-400" />
                      <h3 className="text-lg font-semibold text-white">Explicação</h3>
                    </div>
                    <p className="text-gray-300 leading-relaxed">{analysisResult.explicacao}</p>
                  </div>

                  <div className="space-y-4">
                    <div className="flex items-center space-x-2">
                      <Clock className="w-5 h-5 text-yellow-400" />
                      <h3 className="text-lg font-semibold text-white">Horário de Entrada</h3>
                    </div>
                    <div className="bg-gray-800/50 rounded-lg p-4">
                      {/* ⬇️ Usa o horário convertido para local */}
                      <p className="text-2xl font-mono font-bold text-yellow-400">{entradaLocal}</p>
                      <p className="text-gray-400 text-sm mt-1">Horário recomendado para entrada</p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Action Buttons */}
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button
                onClick={() => router.back()}
                className="bg-gray-700 hover:bg-gray-600 text-white font-semibold px-8 py-3 rounded-xl"
              >
                Nova Análise
              </Button>
              <Button
                onClick={() => router.push("/dashboard")}
                className="bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 text-white font-semibold px-8 py-3 rounded-xl"
              >
                Voltar ao Dashboard
              </Button>
            </div>
          </div>
        </main>
      </div>
    </PageTransition>
  )
}
