"use client"

import type React from "react"
import { useEffect, useRef, useState } from "react"
import { useRouter } from "next/navigation"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { ArrowLeft, TrendingDown, TrendingUp, Loader2 } from "lucide-react"
import Image from "next/image"
import { PageTransition } from "@/components/page-transition"
import LiveChart, {
  LiveChartHandle,
  LiveChartStats,
} from "@/components/LiveChart"

interface AnalysisResult {
  summary: string
  keyPoints: string[]
}

export default function AnalysisPage() {
  const router = useRouter()
  const [analysisResult, setAnalysisResult] = useState<AnalysisResult | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  // Params
  const [symbol, setSymbol] = useState<string | null>(null)
  const [name, setName] = useState<string | null>(null)
  const [imageUrl, setImageUrl] = useState<string | null>(null)
  const [model, setModel] = useState<string | null>(null)

  // Stats vindas do componente
  const [currentPrice, setCurrentPrice] = useState("0")
  const [previousPrice, setPreviousPrice] = useState<string | null>(null)
  const [wsStatus, setWsStatus] = useState<"connecting" | "connected" | "disconnected">("connecting")

  const chartRef = useRef<LiveChartHandle>(null)
  const [isAnalyzing, setIsAnalyzing] = useState(false)

  // Ler params + resultado salvo
  useEffect(() => {
    const storedParams = localStorage.getItem("aiOperationParams")
    if (storedParams) {
      const params = JSON.parse(storedParams)
      setSymbol(params.symbol)
      setName(params.name)
      setImageUrl(params.img)
      setModel(params.model)
    } else {
      router.push("/ai-operation")
    }

    const result = localStorage.getItem("aiAnalysisResult")
    if (result) {
      setAnalysisResult(JSON.parse(result))
      localStorage.removeItem("aiAnalysisResult")
    }
    setIsLoading(false)
  }, [router])

  // Receber stats do componente
  const handleStatsChange = (stats: LiveChartStats) => {
    setCurrentPrice(stats.currentPrice)
    setPreviousPrice(stats.previousPrice)
    setWsStatus(stats.wsStatus)
  }

  const change = (() => {
    const curr = parseFloat(currentPrice || "0")
    const prev = parseFloat((previousPrice ?? currentPrice) || "0")
    const diff = curr - prev
    const pct = prev ? (diff / prev) * 100 : 0
    return { diff, pct }
  })()
  const isPositive = change.diff >= 0

  const handleAutomation = () => {
    localStorage.setItem(
      "aiOperationParams",
      JSON.stringify({ symbol, name, img: imageUrl, model }),
    )
    router.push(`/automation`)
  }

  const handleAnalysis = async () => {
    try {
      setIsAnalyzing(true)
      const chartBase64 = await chartRef.current?.exportAsBase64()
      if (!chartBase64) throw new Error("Failed to export chart")

      const response = await fetch("/api/ai-analysis", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ imageBase64: chartBase64 }),
      })
      if (!response.ok) throw new Error("Failed to get AI analysis")

      const result = await response.json()
      localStorage.setItem("aiAnalysisResult", JSON.stringify(result))
      router.push(`/ai-result`)
    } catch (err) {
      console.error("Error during analysis:", err)
      alert("Erro ao realizar análise. Tente novamente.")
    } finally {
      setIsAnalyzing(false)
    }
  }

  return (
    <PageTransition>
      <div className="min-h-screen bg-gray-950">
        <header className="border-b border-gray-800 bg-gray-900">
          <div className="container mx-auto px-4 py-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <Button
                  variant="ghost"
                  size="icon"
                  className="text-gray-400 hover:text-white hover:bg-gray-800"
                  onClick={() => router.push("/ai-operation")}
                >
                  <ArrowLeft className="w-5 h-5" />
                </Button>
                <div className="flex items-center space-x-3">
                  <div className="p-2 bg-gray-800 rounded-lg">
                    <Image
                      src="/logo-optimus.png"
                      alt="OptimusIA Logo"
                      width={32}
                      height={32}
                      className="drop-shadow-lg"
                    />
                  </div>
                  <span className="text-xl font-bold text-white">OptimusIA</span>
                </div>
              </div>
            </div>
          </div>
        </header>

        <main className="container mx-auto px-4 py-4 md:py-8">
          {/* Asset Header */}
          <div className="mb-6 md:mb-8">
            <div className="flex items-center space-x-3 md:space-x-4 mb-4">
              <div className="flex items-center justify-center w-12 h-12 md:w-16 md:h-16 rounded-xl bg-gray-800 overflow-hidden">
                {imageUrl ? (
                  <img
                    src={imageUrl || "/placeholder.svg"}
                    alt={name || ""}
                    className="w-8 h-8 md:w-12 md:h-12"
                  />
                ) : (
                  <span className="text-lg md:text-xl font-bold text-white">
                    {symbol?.substring(0, 2)}
                  </span>
                )}
              </div>
              <div>
                <h1 className="text-2xl md:text-3xl font-bold text-white">{name}</h1>
                <p className="text-gray-400 text-sm md:text-base">{symbol}</p>
              </div>
            </div>

            <div className="flex flex-col md:flex-row md:items-center space-y-2 md:space-y-0 md:space-x-6">
              <div>
                <p className="text-2xl md:text-3xl font-bold text-white">${currentPrice}</p>
              </div>
              <div className={`flex items-center space-x-1 ${isPositive ? "text-green-400" : "text-red-400"}`}>
                {isPositive ? <TrendingUp className="w-4 h-4 md:w-5 md:h-5" /> : <TrendingDown className="w-4 h-4 md:w-5 md:h-5" />}
                <span className="font-semibold text-sm md:text-base">
                  {isPositive ? "+" : ""}
                  {change.diff.toFixed(2)} ({isPositive ? "+" : ""}
                  {change.pct.toFixed(2)}%)
                </span>
              </div>
            </div>
          </div>

          {/* Chart (componente) */}
          {symbol && (
            <div className="mb-6 md:mb-8">
              <LiveChart
                ref={chartRef}
                symbol={symbol}
                onStatsChange={handleStatsChange}
              />
            </div>
          )}

          {/* AI Analysis Section */}
          <Card className="bg-gray-900 border-gray-800 shadow-lg">
            <CardContent className="p-4 md:p-6">
              <h2 className="text-lg md:text-xl font-semibold text-white mb-4">
                Análise com {model?.toUpperCase()}
              </h2>
              <div className="bg-gradient-to-br from-gray-800 via-gray-800 to-gray-700 rounded-xl p-6 border border-gray-700/50">
                <div className="flex flex-col sm:flex-row gap-6 justify-center">
                  <Button
                    onClick={handleAnalysis}
                    disabled={isAnalyzing || wsStatus !== "connected"}
                    className="group relative bg-gradient-to-r from-blue-600 to-blue-500 hover:from-blue-700 hover:to-blue-600 text-white font-semibold px-8 py-4 rounded-xl shadow-lg hover:shadow-blue-500/25 transition-all duration-300 transform hover:scale-105 border border-blue-500/20 disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none"
                  >
                    <div className="absolute inset-0 bg-gradient-to-r from-blue-400/20 to-transparent rounded-xl opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    <span className="relative flex items-center space-x-2">
                      {isAnalyzing ? (
                        <Loader2 className="w-5 h-5 animate-spin" />
                      ) : (
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                        </svg>
                      )}
                      <span>{isAnalyzing ? "Analisando..." : "Pedir Análise"}</span>
                    </span>
                  </Button>

                  <Button
                    onClick={handleAutomation}
                    className="group relative bg-gradient-to-r from-green-600 to-emerald-500 hover:from-green-700 hover:to-emerald-600 text-white font-semibold px-8 py-4 rounded-xl shadow-lg hover:shadow-green-500/25 transition-all duration-300 transform hover:scale-105 border border-green-500/20"
                  >
                    <div className="absolute inset-0 bg-gradient-to-r from-green-400/20 to-transparent rounded-xl opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    <span className="relative flex items-center space-x-2">
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                      </svg>
                      <span>Automação</span>
                    </span>
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </main>
      </div>
    </PageTransition>
  )
}
