import { type NextRequest, NextResponse } from "next/server"

export const runtime = "nodejs"
export const dynamic = "force-dynamic"
export const fetchCache = "force-no-store"

// Interface para o resultado da análise
interface AnalysisResult {
  recomendacao: string
  probabilidade?: string
  explicacao?: string
  entrada?: string // "HH:mm:ss" em UTC
}

// ===== Util =====
function utcTimePlus1Minute(): string {
  return new Date(Date.now() + 60_000).toISOString().substring(11, 19) // "HH:mm:ss" UTC
}

function buildFallback(mode: "manual" | "automation"): AnalysisResult {
  const recomendacao = Math.random() > 0.5 ? "COMPRA" : "VENDA"
  if (mode === "automation") {
    return { recomendacao }
  }
  const prob = `${Math.floor(Math.random() * (90 - 75 + 1)) + 75}%`
  return {
    recomendacao,
    probabilidade: prob,
    explicacao: "Análise baseada nos padrões de velas recentes e indicadores técnicos do gráfico.",
    entrada: utcTimePlus1Minute(),
  }
}

// Tenta extrair JSON de um conteúdo de texto qualquer
function tryParseJsonFromContent(content: string): any | null {
  try { return JSON.parse(content) } catch {}
  const jsonFence = content.match(/```json\s*([\s\S]*?)\s*```/i)
  if (jsonFence) { try { return JSON.parse(jsonFence[1]) } catch {} }
  const firstObj = content.match(/\{[\s\S]*\}/)
  if (firstObj) { try { return JSON.parse(firstObj[0]) } catch {} }
  return null
}

// ===== Chamada ao OpenAI com tolerância a falhas =====
async function callOpenAI(imageBase64: string, mode: "manual" | "automation"): Promise<AnalysisResult> {
  const fallbackResponse = buildFallback(mode)
  const randomProbability = Math.floor(Math.random() * (90 - 75 + 1)) + 75
  const entryUtc = utcTimePlus1Minute()

  const prompt =
    mode === "automation"
      ? `IMPORTANT: You are a professional trader tasked with analyzing chart images and providing structured recommendations.
IMPORTANT: You must ONLY return a valid JSON object with no additional text before or after it.
Analyze the chart image where candles last 15 minutes.

Return EXACTLY this JSON:
{"recomendacao":"COMPRA"|"VENDA"}

Faça apenas o que estou te pedindo, por favor!`
      : `IMPORTANT: You are a professional trader tasked with analyzing chart images and providing structured recommendations.
IMPORTANT: You must ONLY return a valid JSON object with no additional text before or after it.
Analyze the chart image where candles last 15 minutes.

Return EXACTLY this JSON:
{
  "recomendacao":"COMPRA"|"VENDA",
  "probabilidade":"${randomProbability}%",
  "explicacao":"A technical explanation in Portuguese",
  "entrada":"${entryUtc}"
}

Faça apenas o que estou te pedindo, por favor!`

  const maxRetries = 2
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      const controller = new AbortController()
      const timeoutId = setTimeout(() => controller.abort(), 8000)

      const apiKey = process.env.OPENAI_API_KEY
      if (!apiKey) return fallbackResponse

      const resp = await fetch("https://api.openai.com/v1/chat/completions", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${apiKey}`,
        },
        body: JSON.stringify({
          model: "gpt-4o",
          messages: [
            {
              role: "system",
              content:
                "You are a professional trader. Only return a valid JSON object. No extra text before or after the JSON.",
            },
            {
              role: "user",
              content: [
                { type: "text", text: prompt },
                { type: "image_url", image_url: { url: imageBase64 } },
              ],
            },
          ],
          max_tokens: mode === "automation" ? 50 : 300,
          temperature: 0.7,
        }),
        signal: controller.signal,
      })

      clearTimeout(timeoutId)

      if (!resp.ok) continue

      const data = await resp.json()
      const content: string | undefined = data?.choices?.[0]?.message?.content
      if (!content) continue

      const jsonData = tryParseJsonFromContent(content)
      if (!jsonData) continue

      const recomendacao = String(jsonData.recomendacao ?? jsonData.recomendação ?? fallbackResponse.recomendacao).toUpperCase()

      if (mode === "automation") {
        return { recomendacao }
      } else {
        return {
          recomendacao,
          probabilidade: `${randomProbability}%`,
          explicacao: String(jsonData.explicacao ?? jsonData.explicação ?? fallbackResponse.explicacao ?? ""),
          entrada: entryUtc, // sempre UTC
        }
      }
    } catch {
      continue
    }
  }
  return fallbackResponse
}

export async function POST(request: NextRequest) {
  try {
    let body: { imageBase64?: string; mode?: string } = {}
    try {
      body = await request.json()
    } catch {
      return NextResponse.json({ error: "Erro ao processar a requisição" }, { status: 400 })
    }

    const { imageBase64, mode = "manual" } = body
    if (!imageBase64) {
      return NextResponse.json({ error: "Imagem não fornecida" }, { status: 400 })
    }

    const modeSafe: "manual" | "automation" = mode === "automation" ? "automation" : "manual"
    const result = await callOpenAI(imageBase64, modeSafe)
    return NextResponse.json(result)
  } catch {
    return NextResponse.json(buildFallback("manual"))
  }
}
