// /var/www/html/optimus/app/api/auth/login/route.ts
export const runtime = "nodejs";

import { type NextRequest, NextResponse } from "next/server";
import mysql from "mysql2/promise";

// Pool usando o .env da raiz do projeto (Next carrega automaticamente)
const pool = mysql.createPool({
  host: process.env.DB_HOST || "127.0.0.1",
  port: Number(process.env.DB_PORT || 3306),
  user: process.env.DB_USER,
  password: process.env.DB_PASS,
  database: process.env.DB_NAME,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0,
});

export async function POST(request: NextRequest) {
  try {
    const { email, password } = await request.json();

    if (!email || !password) {
      return NextResponse.json(
        { success: false, message: "Email e senha são obrigatórios" },
        { status: 400 }
      );
    }

    console.log("=== TENTATIVA DE LOGIN VIA PROXY ===");
    console.log("User:", email);
    console.log("URL do Proxy:", "https://web-taurus.com/login");

    // Criar FormData para o proxy
    const formData = new URLSearchParams();
    formData.append("user", email);
    formData.append("pass", password);

    console.log("Enviando dados para proxy:", { user: email, pass: "***" });

    // Fazer login via proxy
    const response = await fetch("https://web-taurus.com/ws/login", {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Content-Type": "application/x-www-form-urlencoded",
      },
      body: formData,
    });

    console.log("=== RESPOSTA DO PROXY ===");
    console.log("Status:", response.status);
    console.log("Status Text:", response.statusText);

    const responseText = await response.text();
    console.log("Resposta bruta do proxy:", responseText);

    let data: any;
    try {
      data = JSON.parse(responseText);
      console.log("Dados parseados do proxy:", data);
    } catch (parseError) {
      console.error("Erro ao parsear JSON do proxy:", parseError);
      return NextResponse.json(
        {
          success: false,
          message: "Resposta inválida do servidor proxy",
          debug: responseText,
        },
        { status: 500 }
      );
    }

    if (!response.ok) {
      console.log("=== ERRO NA RESPOSTA DO PROXY ===");
      console.log("Status:", response.status);
      console.log("Dados:", data);

      return NextResponse.json(
        {
          success: false,
          message: data.msg || data.message || "Credenciais inválidas",
          debug: data,
        },
        { status: response.status }
      );
    }

    if (data.status !== "success") {
      console.log("=== LOGIN NÃO FOI BEM-SUCEDIDO VIA PROXY ===");
      console.log("Status:", data.status);
      console.log("Dados completos:", data);

      return NextResponse.json(
        {
          success: false,
          message: data.msg || data.message || "Credenciais inválidas",
          debug: data,
        },
        { status: 401 }
      );
    }

    if (!data.token) {
      console.log("=== TOKEN NÃO ENCONTRADO NA RESPOSTA DO PROXY ===");
      console.log("Dados completos:", data);

      return NextResponse.json(
        {
          success: false,
          message: "Token de acesso não retornado pelo proxy",
          debug: data,
        },
        { status: 500 }
      );
    }

    console.log("=== LOGIN BEM-SUCEDIDO VIA PROXY ===");
    console.log("Token:", String(data.token).substring(0, 20) + "...");
    console.log("User ID:", data.id);
    console.log("Name:", data.name);
    console.log("Credit:", data.credit);

    // ====== SALVAR/ATUALIZAR PELO LOGIN ======
    const loginKey = email;
    const taurusId = Number(data.id) || null;
    const credit   = data.credit ?? null;
    const token    = String(data.token);
    const nameVal  = (data.name && String(data.name).trim()) || email; // garante valor para INSERT

    try {
      // 1) Tenta atualizar registro existente (name só se estiver vazio/NULL)
      const [updResult] = await pool.execute(
        `UPDATE user_registrations
           SET taurus_id = ?, credit = ?, token = ?,
               name = IF(name IS NULL OR name = '', ?, name)
         WHERE login = ?`,
        [taurusId, credit, token, nameVal, loginKey]
      );

      const affected = (updResult as any)?.affectedRows ?? 0;

      // 2) Se não havia linha com esse login, insere uma nova
      if (affected === 0) {
        await pool.execute(
          `INSERT INTO user_registrations
            (name, login, email, password, taurus_id, credit, token, registered_at)
           VALUES (?,?,?,?,?,?,?, NOW())`,
          [
            nameVal,   // name
            loginKey,  // login
            loginKey,  // email (usa o próprio login)
            "",        // password vazio para cumprir NOT NULL
            taurusId,  // taurus_id
            credit,    // credit
            token      // token
          ]
        );
      }
    } catch (e: any) {
      console.error("⚠️ UPSERT em user_registrations falhou:", e?.message || e);
      // não bloqueia o retorno do login
    }

    // Resposta ao cliente
    return NextResponse.json({
      success: true,
      token: data.token,
      user: {
        id: data.id,
        name: data.name,
        credit: data.credit,
      },
    });
  } catch (error) {
    console.error("=== ERRO GERAL NO LOGIN VIA PROXY ===", error);
    return NextResponse.json(
      {
        success: false,
        message:
          "Erro ao conectar com o servidor proxy. Verifique se está rodando na porta 3030.",
        error: error instanceof Error ? error.message : "Erro desconhecido",
      },
      { status: 500 }
    );
  }
}
