// app/api/transaction/route.ts
import { type NextRequest, NextResponse } from "next/server"

export const runtime = "nodejs"
export const dynamic = "force-dynamic"
export const fetchCache = "force-no-store"

export async function POST(request: NextRequest) {
  try {
    const { expiration, amount, direction, symbol, symbolPrice } = await request.json()

    // validação básica
    if (!expiration || !amount || !direction || !symbol || !symbolPrice) {
      return NextResponse.json({ error: "MISSING_FIELDS" }, { status: 400 })
    }

    const authHeader = request.headers.get("authorization") || request.headers.get("Authorization")
    if (!authHeader?.startsWith("Bearer ")) {
      return NextResponse.json({ error: "Token required" }, { status: 401 })
    }
    const token = authHeader.slice(7).trim()

    // monta body x-www-form-urlencoded
    const formData = new URLSearchParams()
    formData.set("expiration", String(expiration))
    formData.set("amount", String(amount))
    formData.set("direction", String(direction))
    formData.set("symbol", String(symbol))
    formData.set("symbol_price", String(symbolPrice))

    const upstream = await fetch("https://web-taurus.com/ws/transaction", {
      method: "PUT",
      headers: {
        Authorization: `Bearer ${token}`,
        "Content-Type": "application/x-www-form-urlencoded",
      },
      body: formData.toString(),
      cache: "no-store",
    })

    // propaga exatamente o status/erro do provedor
    if (!upstream.ok) {
      const detailText = await upstream.text().catch(() => "")
      // retorna 401 quando o provedor devolve 401 (não transforma em 500)
      return NextResponse.json(
        {
          error: upstream.status === 401 ? "BROKER_401" : "BROKER_ERROR",
          status: upstream.status,
          detail: detailText || undefined,
        },
        { status: upstream.status },
      )
    }

    const data = await upstream.json()
    return NextResponse.json(data)
  } catch (err: any) {
    // erro inesperado local (rede, parse, etc)
    console.error("transaction route error:", err?.message || err)
    return NextResponse.json({ error: "INTERNAL", message: err?.message || "Unknown error" }, { status: 500 })
  }
}
