import { type NextRequest, NextResponse } from "next/server"
import axios from "axios"
import https from "https"
import dns from "dns"

export const runtime = "nodejs"
export const dynamic = "force-dynamic"
// evita qualquer cache de fetch do Next em edge
export const fetchCache = "force-no-store"

export async function GET(request: NextRequest) {
  try {
    const authHeader = request.headers.get("authorization")
    if (!authHeader?.startsWith("Bearer ")) {
      return NextResponse.json({ message: "Token não fornecido ou formato inválido" }, { status: 401 })
    }
    const token = authHeader.slice(7).trim()

    // Permite configurar via .env.local, mantendo defaults que você já usa
    const SCHEME = process.env.USER_BALANCE_SCHEME ?? "https"
    const HOST = process.env.USER_BALANCE_HOST ?? "taurusbroker.com"
    const PATH = process.env.USER_BALANCE_PATH ?? "/api/public/users/balance"

    // 1) Pré-resolve IPv4
    let targetUrl = `${SCHEME}://${HOST}${PATH}`
    let agent = new https.Agent({ keepAlive: true }) // fallback (hostname normal)

    try {
      const [ipv4] = await dns.promises.resolve4(HOST) // força A record
      if (ipv4) {
        // 2) Chama via IP, preservando SNI e Host (TLS + Cloudflare ok)
        targetUrl = `${SCHEME}://${ipv4}${PATH}`
        agent = new https.Agent({
          keepAlive: true,
          servername: HOST, // SNI correto
        })
      }
    } catch {
      // Se não resolver A record, segue pelo hostname mesmo
    }

    const r = await axios.get(targetUrl, {
      headers: {
        Authorization: `Bearer ${token}`,
        Accept: "application/json",
        Host: HOST, // necessário quando usa IP
        "User-Agent":
          "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119 Safari/537.36",
      },
      httpsAgent: agent,
      validateStatus: (s) => s < 500, // tratamos 4xx manualmente; 5xx vira catch
      timeout: 15000,
      responseType: "text",        // queremos ver HTML cru se CF bloquear
      transformResponse: (v) => v, // sem parse automático
    })

    const ct = (r.headers["content-type"] || "").toLowerCase()
    const fromCF = (r.headers["server"] || "").toString().toLowerCase().includes("cloudflare")
    const looksHtml = ct.includes("text/html") || /<html/i.test(r.data as string)

    if (fromCF && looksHtml) {
      return NextResponse.json(
        {
          message: "Bloqueado pela Cloudflare (403). Peça allowlist/rota de API sem challenge ao provedor.",
          ray: r.headers["cf-ray"] || null,
        },
        { status: 403 }
      )
    }

    if (r.status !== 200) {
      return NextResponse.json(
        { message: `Erro ao buscar dados do usuário: ${r.status}`, details: (r.data as string)?.slice?.(0, 2000) },
        { status: r.status }
      )
    }

    let data: any
    try {
      data = JSON.parse(r.data as string)
    } catch {
      return NextResponse.json({ message: "Resposta inválida do provedor." }, { status: 502 })
    }

    // Retorna exatamente o que o provedor enviou (mantém compatibilidade com seu front)
    return NextResponse.json(data)
  } catch (error: any) {
    // Se cair aqui é 5xx (timeout, rede, TLS, etc.)
    console.error("Erro ao buscar saldo:", error?.message || error)
    return NextResponse.json({ message: "Erro ao conectar com o servidor. Tente novamente." }, { status: 500 })
  }
}
