"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Loader2, Shield, Zap, Bot, AlertCircle } from "lucide-react"
import Image from "next/image"
import { PageTransition } from "@/components/page-transition"

export default function LoginPage() {
  const [email, setEmail] = useState("")
  const [password, setPassword] = useState("")
  const [isLoading, setIsLoading] = useState(false)
  const [error, setError] = useState("")
  const [debugInfo, setDebugInfo] = useState<any>(null)
  const router = useRouter()

  useEffect(() => {
    // Verificar se já existe um token válido
    const token = localStorage.getItem("token")
    if (token) {
      router.push("/dashboard")
    }
  }, [router])

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)
    setError("")
    setDebugInfo(null)

    console.log("=== INICIANDO LOGIN ===")
    console.log("User:", email)

    try {
      const response = await fetch("/api/auth/login", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ email, password }),
      })

      console.log("=== RESPOSTA DO ENDPOINT LOCAL ===")
      console.log("Status:", response.status)

      // --- SAFE PARSE (handles text or JSON) -----------------
      let data: any
      const raw = await response.text()

      try {
        data = JSON.parse(raw) // API returned valid JSON
      } catch {
        data = { message: raw } // Fallback for plain-text/HTML errors
      }
      // --------------------------------------------------------

      console.log("Dados recebidos:", data)

      if (response.ok && data.success) {
        console.log("=== LOGIN BEM-SUCEDIDO ===")
        localStorage.setItem("token", data.token)
        localStorage.setItem("user", JSON.stringify(data.user))
        router.push("/dashboard")
      } else {
        console.log("=== ERRO NO LOGIN ===")
        setError(data.message || "Erro ao fazer login")
        setDebugInfo(data.debug || data)
      }
    } catch (error) {
      console.error("=== ERRO DE CONEXÃO ===", error)
      setError("Erro de conexão. Tente novamente.")
    } finally {
      setIsLoading(false)
    }
  }

  return (
    <PageTransition>
      <div className="min-h-screen bg-gradient-to-br from-gray-950 via-slate-900 to-gray-950 flex items-center justify-center p-4">
        {/* Background Effects */}
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute -top-40 -right-40 w-80 h-80 bg-blue-600/10 rounded-full mix-blend-multiply filter blur-xl animate-pulse"></div>
          <div className="absolute -bottom-40 -left-40 w-80 h-80 bg-cyan-500/10 rounded-full mix-blend-multiply filter blur-xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-80 h-80 bg-indigo-600/10 rounded-full mix-blend-multiply filter blur-xl animate-pulse delay-500"></div>
        </div>

        <div className="w-full max-w-md relative z-10">
          <div className="text-center mb-8">
            <div className="inline-flex items-center justify-center mb-6 p-4 bg-gray-900/50 rounded-2xl backdrop-blur-sm border border-gray-800">
              <Image
                src="/logo-optimus.png"
                alt="OptimusIA Logo"
                width={120}
                height={120}
                className="drop-shadow-2xl"
              />
            </div>
            <h1 className="text-3xl font-bold text-white mb-2">OptimusIA</h1>
            <p className="text-gray-400">Inteligência Artificial para Trading Profissional</p>
          </div>

          <Card className="bg-gray-900/80 border-gray-800 backdrop-blur-xl shadow-2xl">
            <CardHeader className="space-y-1">
              <CardTitle className="text-2xl text-center text-white">Entrar na conta</CardTitle>
              <CardDescription className="text-center text-gray-400">
                Digite suas credenciais para acessar sua conta
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleLogin} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="email" className="text-gray-200">
                    Usuário
                  </Label>
                  <Input
                    id="email"
                    type="text"
                    placeholder="seu@email.com"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    required
                    className="bg-gray-800/80 border-gray-700 text-white placeholder:text-gray-500 focus:border-blue-500 focus:ring-blue-500/20"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="password" className="text-gray-200">
                    Senha
                  </Label>
                  <Input
                    id="password"
                    type="password"
                    placeholder="••••••••"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    required
                    className="bg-gray-800/80 border-gray-700 text-white placeholder:text-gray-500 focus:border-blue-500 focus:ring-blue-500/20"
                  />
                </div>

                {error && (
                  <Alert className="bg-red-500/10 border-red-500/30 text-red-400">
                    <AlertCircle className="h-4 w-4" />
                    <AlertDescription>{error}</AlertDescription>
                  </Alert>
                )}

                {/* Debug Info */}
                {debugInfo && (
                  <Alert className="bg-yellow-500/10 border-yellow-500/30 text-yellow-400">
                    <AlertCircle className="h-4 w-4" />
                    <AlertDescription>
                      <details className="mt-2">
                        <summary className="cursor-pointer">Informações de Debug</summary>
                        <pre className="text-xs mt-2 overflow-auto bg-gray-950/50 p-2 rounded">
                          {JSON.stringify(debugInfo, null, 2)}
                        </pre>
                      </details>
                    </AlertDescription>
                  </Alert>
                )}

                <Button
                  type="submit"
                  className="w-full bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 text-white font-semibold py-2.5 shadow-lg hover:shadow-blue-500/25 transition-all duration-300"
                  disabled={isLoading}
                >
                  {isLoading ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Entrando...
                    </>
                  ) : (
                    "Entrar"
                  )}
                </Button>
              </form>

              <div className="mt-6 pt-6 border-t border-gray-800">
                <div className="grid grid-cols-3 gap-4 text-center">
                  <div className="flex flex-col items-center space-y-2">
                    <Shield className="w-6 h-6 text-green-400" />
                    <span className="text-xs text-gray-400">Seguro</span>
                  </div>
                  <div className="flex flex-col items-center space-y-2">
                    <Zap className="w-6 h-6 text-yellow-400" />
                    <span className="text-xs text-gray-400">Rápido</span>
                  </div>
                  <div className="flex flex-col items-center space-y-2">
                    <Bot className="w-6 h-6 text-blue-400" />
                    <span className="text-xs text-gray-400">IA Avançada</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <p className="text-center text-gray-400 text-sm mt-6">
            Não tem uma conta?{" "}
            <a href="#" className="text-blue-400 hover:text-blue-300 font-medium transition-colors">
              Entre em contato
            </a>
          </p>
        </div>
      </div>
    </PageTransition>
  )
}
