import type BaseClient from 'common/lib/client/baseclient';
import type RealtimeChannel from 'common/lib/client/realtimechannel';
import { ObjectMessage, ObjectState } from './objectmessage';
import { Objects } from './objects';

export interface LiveObjectDataEntry {
  objectState: ObjectState;
  objectType: 'LiveMap' | 'LiveCounter';
}

export interface LiveCounterDataEntry extends LiveObjectDataEntry {
  objectType: 'LiveCounter';
}

export interface LiveMapDataEntry extends LiveObjectDataEntry {
  objectType: 'LiveMap';
}

export type AnyDataEntry = LiveCounterDataEntry | LiveMapDataEntry;

// TODO: investigate if this class is still needed after changes with createOp. objects are now initialized from the stateObject and this class does minimal processing
/**
 * @internal
 */
export class SyncObjectsDataPool {
  private _client: BaseClient;
  private _channel: RealtimeChannel;
  private _pool: Map<string, AnyDataEntry>;

  constructor(private _objects: Objects) {
    this._client = this._objects.getClient();
    this._channel = this._objects.getChannel();
    this._pool = new Map<string, AnyDataEntry>();
  }

  entries() {
    return this._pool.entries();
  }

  size(): number {
    return this._pool.size;
  }

  isEmpty(): boolean {
    return this._pool.size === 0;
  }

  clear(): void {
    this._pool.clear();
  }

  applyObjectSyncMessages(objectMessages: ObjectMessage[]): void {
    for (const objectMessage of objectMessages) {
      if (!objectMessage.object) {
        this._client.Logger.logAction(
          this._client.logger,
          this._client.Logger.LOG_MAJOR,
          'SyncObjectsDataPool.applyObjectSyncMessages()',
          `object message is received during OBJECT_SYNC without 'object' field, skipping message; message id: ${objectMessage.id}, channel: ${this._channel.name}`,
        );
        continue;
      }

      const objectState = objectMessage.object;

      if (objectState.counter) {
        this._pool.set(objectState.objectId, this._createLiveCounterDataEntry(objectState));
      } else if (objectState.map) {
        this._pool.set(objectState.objectId, this._createLiveMapDataEntry(objectState));
      } else {
        this._client.Logger.logAction(
          this._client.logger,
          this._client.Logger.LOG_MAJOR,
          'SyncObjectsDataPool.applyObjectSyncMessages()',
          `received unsupported object state message during OBJECT_SYNC, expected 'counter' or 'map' to be present, skipping message; message id: ${objectMessage.id}, channel: ${this._channel.name}`,
        );
      }
    }
  }

  private _createLiveCounterDataEntry(objectState: ObjectState): LiveCounterDataEntry {
    const newEntry: LiveCounterDataEntry = {
      objectState,
      objectType: 'LiveCounter',
    };

    return newEntry;
  }

  private _createLiveMapDataEntry(objectState: ObjectState): LiveMapDataEntry {
    const newEntry: LiveMapDataEntry = {
      objectState,
      objectType: 'LiveMap',
    };

    return newEntry;
  }
}
