// Base de dados simulada (em produção, use um banco real)
let users = [];
let affiliateData = [];

/**********************************************************************
 * SERVIDOR SIMPLIFICADO – SOMENTE CADASTRO E AFILIADOS
 * - Sistema de afiliados com captura de tokens
 * - Registro de usuários → SEU PROXY
 *********************************************************************/
import dotenv from 'dotenv';
dotenv.config();
import express from 'express';
import axios   from 'axios';
import path    from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import { pool } from './db.js'; // <-- ADICIONADO

const __filename = fileURLToPath(import.meta.url);
const __dirname  = dirname(__filename);

/*────────── Config --------------------------------------------------*/
const PORT      = process.env.PORT || 3030;
const PROXY_API = 'https://web-taurus.com/ws/users/register';   // seu proxy

/*────────── Express -------------------------------------------------*/
const app = express();
app.use(express.json());                         // aceita JSON do front
app.use(express.static(path.join(__dirname, '../frontend')));

/* CORS simples */
app.use((req, res, next) => {
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'GET,POST,OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type, Authorization');
  if (req.method === 'OPTIONS') return res.sendStatus(204);
  next();
});

/*════════════════════════════════════════════════════════════════════
 * SISTEMA DE AFILIADOS (sem alterações)
 ════════════════════════════════════════════════════════════════════*/
app.get('/api/affiliate/capture/:affiliateId/:campaignId', async (req, res) => {
  const { affiliateId, campaignId } = req.params;
  console.log(`🔗 Capturando afiliado: ${affiliateId}/${campaignId}`);

  try {
    const staking = `https://stke.me/l/${affiliateId}/${campaignId}`;
    const r = await axios.get(staking, {
      maxRedirects: 0,
      validateStatus: s => s === 302 || s < 400,
      timeout: 10000
    });

    if (r.status === 302 && r.headers.location?.includes('ztoken=')) {
      const token = r.headers.location.match(/ztoken=([^&\r\n]+)/)?.[1];
      if (!token) throw new Error('ztoken não encontrado');

      const info = {
        token, affiliateId, campaignId,
        originalRedirect: r.headers.location,
        capturedAt: new Date().toISOString(),
        status: 'success'
      };
      affiliateData.push(info);
      console.log('✅ Token capturado:', token);
      return res.json(info);
    }
    throw new Error(`Redirecionamento inesperado (${r.status})`);
  } catch (err) {
    console.error('❌ Capture error:', err.message);
    res.status(500).json({ success:false, error:err.message, status:'error' });
  }
});

/*════════════════════════════════════════════════════════════════════
 * REGISTRO DE USUÁRIO  →  SEU PROXY + SALVAR NO MySQL (SOMENTE CAMPOS PEDIDOS)
 ════════════════════════════════════════════════════════════════════*/
app.post('/api/users/register', async (req, res) => {
  const u = req.body;                    // vindo do front em JSON
  console.log('👤 Novo registro:', u);

  if (!u.name || !u.email)
    return res.status(400).json({ success:false, error:'Nome e e-mail obrigatórios' });

  /* Objeto que o seu proxy espera em JSON -------------------------- */
  const payload = {
    name:     u.name,
    document: u.document  || '089.408.216-70',
    phone:    u.phone     || '(99) 99999-9999',
    email:    u.email,
    login:    u.login     || u.email.split('@')[0] + Math.random().toString(36).slice(2,4),
    password: u.password  || '123456',
    rpassword:u.rpassword || u.password || '123456',
    affiliate_token: u.affiliate_token || null   // mantido, porém não vai para o DB
  };

  try {
    const proxyRes = await axios.post(PROXY_API, payload, {
      headers: { 'Content-Type': 'application/json' },
      timeout: 15000
    });

    console.log('📥 Resposta do proxy:', proxyRes.data);

    if (proxyRes.data.status !== 'success')
      throw new Error(proxyRes.data.msg || 'Falha no proxy');

    /* Histórico em memória (mantido) ------------------------------- */
    users.push({
      id: users.length + 1,
      taurusId:    proxyRes.data.user.id,
      taurusLogin: proxyRes.data.user.login,
      name:  u.name,
      email: u.email,
      phone: u.phone || null,
      affiliate: u.affiliate_token ? {
        token: u.affiliate_token,
        affiliateId: u.affiliate_id,
        campaignId:  u.campaign_id
      } : null,
      registeredAt: new Date().toISOString()
    });

    /* === SALVAR NO MySQL: SOMENTE name, document, phone, email, login, password, registered_at === */
    try {
      await pool.execute(
        `INSERT INTO user_registrations
          (name, document, phone, email, login, password, registered_at)
         VALUES (?,?,?,?,?,?, NOW())`,
        [
          payload.name,
          payload.document || null,
          payload.phone || null,
          payload.email,
          payload.login,
          payload.password // em texto puro, conforme solicitado
        ]
      );
    } catch (dbErr) {
      console.error('⚠️ Erro ao salvar no MySQL:', dbErr.message);
      // não quebra o fluxo se o insert falhar
    }
    /* ================================================================================================= */

    res.json({ success:true, data: proxyRes.data });
  } catch (err) {
    console.error('❌ Registro erro:', err.response?.data || err.message);
    const r = err.response?.data;
    res.status(err.response?.status || 500).json({
      success:false,
      error: r?.msg || r?.message || err.message,
      details: r || null
    });
  }
});

/*════════════════════════════════════════════════════════════════════
 * ROTAS FRONTEND
 ════════════════════════════════════════════════════════════════════*/
app.get('/signup', (_q, res) => res.sendFile(path.join(__dirname, '../frontend/signup.html')));
app.get('/aff/:affiliateId/:campaignId', (req, res) => {
  const { affiliateId, campaignId } = req.params;
  res.redirect(`/signup?aff=${affiliateId}&camp=${campaignId}`);
});

/*════════════════════════════════════════════════════════════════════
 * START
 ════════════════════════════════════════════════════════════════════*/
app.listen(PORT, () => {
  console.log(`
🚀 BACK-END DE CADASTRO em http://localhost:${PORT}

🏠 Páginas:
GET /signup                           – formulário
GET /aff/:affiliate/:campaign         – link de afiliado

🔗 Afiliados:
GET /api/affiliate/capture/:aff/:camp – captura ztoken

👤 Cadastro:
POST /api/users/register              – envia JSON → seu proxy + salva MySQL (somente campos solicitados)
`);
});
