"use client"

import { Badge } from "@/components/ui/badge"

interface AssetItemProps {
  symbol: string
  name: string
  type: string
  imageUrl?: string
  isActive?: boolean
  onClick?: () => void
  isSelected?: boolean
}

export function AssetItem({
  symbol,
  name,
  type,
  imageUrl,
  isActive = true,
  onClick,
  isSelected = false,
}: AssetItemProps) {
  // Generate a consistent color based on the symbol name
  const getSymbolColor = (symbolStr: string) => {
    // Default color if symbol is undefined or empty
    if (!symbolStr) return "blue"

    const colors = ["blue", "green", "yellow", "red", "purple", "orange"]
    const index = symbolStr.split("").reduce((acc, char) => acc + char.charCodeAt(0), 0) % colors.length
    return colors[index]
  }

  // Safely get the symbol's first two characters
  const getSymbolShort = (symbolStr: string) => {
    if (!symbolStr) return "XX"
    return symbolStr.substring(0, 2)
  }

  // Format asset type for display
  const formatAssetType = (assetType: string) => {
    if (assetType === "crypto") return "Cripto"
    if (assetType === "forex") return "Moedas"
    if (assetType === "stocks") return "Ações"
    return assetType
  }

  const color = getSymbolColor(symbol)

  // Use hardcoded color classes instead of dynamic ones to avoid Tailwind purge issues
  let bgColorClass = "bg-blue-600/20"
  let textColorClass = "text-blue-400"

  if (color === "green") {
    bgColorClass = "bg-green-600/20"
    textColorClass = "text-green-400"
  } else if (color === "yellow") {
    bgColorClass = "bg-yellow-600/20"
    textColorClass = "text-yellow-400"
  } else if (color === "red") {
    bgColorClass = "bg-red-600/20"
    textColorClass = "text-red-400"
  } else if (color === "purple") {
    bgColorClass = "bg-purple-600/20"
    textColorClass = "text-purple-400"
  } else if (color === "orange") {
    bgColorClass = "bg-orange-600/20"
    textColorClass = "text-orange-400"
  }

  return (
    <div
      className={`flex items-center justify-between p-4 rounded-lg cursor-pointer transition-all ${
        isSelected ? "bg-blue-600/20 border border-blue-500/50" : "hover:bg-gray-800/50"
      }`}
      onClick={onClick}
    >
      <div className="flex items-center space-x-3">
        <div className="flex items-center justify-center w-10 h-10 rounded-md bg-gray-800 overflow-hidden">
          {imageUrl ? (
            <img src={imageUrl || "/placeholder.svg"} alt={name} className="w-8 h-8" />
          ) : (
            <div className={`flex items-center justify-center w-8 h-8 rounded-md ${bgColorClass}`}>
              <span className={`text-sm font-bold ${textColorClass}`}>{getSymbolShort(symbol)}</span>
            </div>
          )}
        </div>
        <div>
          <p className="font-medium text-white">{name || "Unknown"}</p>
          <p className="text-xs text-gray-400">
            {symbol || ""} • {formatAssetType(type)}
          </p>
        </div>
      </div>
      <div className="flex items-center space-x-3">
        <div className="w-24 h-2 bg-gray-800 rounded-full overflow-hidden">
          <div
            className="h-full bg-blue-500 rounded-full"
            style={{ width: `${Math.floor(Math.random() * 40) + 60}%` }}
          ></div>
        </div>
        <Badge
          variant="outline"
          className={`${isActive ? "bg-green-500/10 text-green-400 border-green-500/30" : "bg-red-500/10 text-red-400 border-red-500/30"}`}
        >
          {isActive ? "Online" : "Offline"}
        </Badge>
      </div>
    </div>
  )
}
