// /var/www/html/optimus/lib/scheduler.ts
// Agendador simples em memória do processo Node (Next runtime: "nodejs")

type Args = {
  transactionId: number
  expirationDatetime: string // ISO
  bearerToken: string
  baseUrl: string            // vem da rota (montado a partir dos headers)
}

const jobs = new Map<number, NodeJS.Timeout>()

function isClosedStatus(status: unknown) {
  const s = String(status ?? "").toLowerCase().trim()
  // cobre variações comuns da sua API/external (PT/EN)
  return [
    "ganhou",
    "perdeu",
    "win",
    "won",
    "loss",
    "lost",
    "closed",
    "settled",
    "finalizado",
    "finalizada",
    "fechada",
  ].includes(s)
}

export function scheduleSettlement({ transactionId, expirationDatetime, bearerToken, baseUrl }: Args) {
  // Se já existir agendamento pra esse ID, cancela e recria
  const existing = jobs.get(transactionId)
  if (existing) clearTimeout(existing)

  const expireAt = new Date(expirationDatetime).getTime()
  const now = Date.now()
  const delay = Math.max(expireAt + 10_000 - now, 1_000) // 10s após expirar (mín. 1s)

  const timeout = setTimeout(async () => {
    try {
      // 1) Verifica status atual
      const checkRes = await fetch(`${baseUrl}/api/transaction/${transactionId}`, {
        method: "GET",
        headers: { Authorization: `Bearer ${bearerToken}` },
        cache: "no-store",
      })

      if (checkRes.ok) {
        const checkData = await checkRes.json()
        const status = checkData?.transaction?.status
        if (isClosedStatus(status)) {
          // Já está fechado — não faz nada
          return
        }
      }

      // 2) Ainda não fechou => roda settlement
      await fetch(`${baseUrl}/api/settlement`, {
        method: "GET",
        headers: { Authorization: `Bearer ${bearerToken}` },
        cache: "no-store",
      })
    } catch (e) {
      console.error("[scheduler] erro ao tentar settlement:", e)
    } finally {
      jobs.delete(transactionId)
    }
  }, delay)

  jobs.set(transactionId, timeout)
  return { scheduledInMs: delay }
}
